<?php

declare(strict_types=1);

namespace FiloBlu\Venistar\Esb\Consumer;

use FiloBlu\Esb\Api\Data\MessageInterface;
use FiloBlu\Esb\Api\Data\StatusInterface;
use Magento\Sales\Api\Data\OrderInterface;
use Throwable;

/**
 *
 */
class SendOrderToVenistar extends SendObjectToFtpConsumer
{

    /**
     * @param \FiloBlu\Esb\Api\Data\MessageInterface $message
     * @return \FiloBlu\Esb\Api\Data\MessageInterface
     * @throws \FiloBlu\Esb\Core\Exception\NonRecoverableException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function consume(MessageInterface $message): MessageInterface
    {
        $order = $this->loadObject($message);

        try {
            $filename = $this->getFileName($order);
            $fileUploaded = $this->uploadFile($filename, $this->createFile($filename, $order));
        } catch (Throwable $throwable) {
            $this->getStatus()
                ->setCode(StatusInterface::ERROR)
                ->appendOutputData(
                    "Can't generate JSON from order {$order->getIncrementId()}\n" . $throwable->getMessage(
                    ) . "\n" . $throwable->getTraceAsString()
                );
            return $this->messageFactory->create();
        }

        if (!$fileUploaded) {
            $this->getStatus()->setCode(StatusInterface::ERROR);
            $this->getStatus()->appendOutputData("File upload on FTP failed for order {$order->getIncrementId()}");
            return $this->messageFactory->create();
        }

        $this->getStatus()->setCode(StatusInterface::SUCCESS);
        $this->getStatus()->appendOutputData("File uploaded on FTP successfully for order {$order->getIncrementId()}");
        return $this->messageFactory->create();
    }

    /**
     * @param \Magento\Sales\Api\Data\OrderInterface $order
     * @return string
     */
    public function getFileName(OrderInterface $order): string
    {
        return sprintf('order_%s_%s.json', $order->getIncrementId(), date('YmdHis'));
    }

    /**
     * @return string
     */
    public function getName(): string
    {
        return 'venistar_order_sender';
    }

    /**
     * @return string
     */
    public function getUniqueName(): string
    {
        return 'FiloBlu_Venistar::order_sender';
    }
}
