<?php

namespace FiloBlu\ProductUrlTools\Console\Command;

use FiloBlu\ProductUrlTools\Helper\Category;
use FiloBlu\ProductUrlTools\Helper\Progress;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

/**
 * Class RegenerateCategoryUrlCommand
 * @package FiloBlu\ProductUrlTools\Console\Command
 */
class RegenerateCategoryUrlCommand extends Command
{
    /**
     * @var Category
     */
    protected $categoryHelper;
    /**
     * @var ResourceConnection
     */
    private $resource;

    /**
     * RegenerateCategoryUrlCommand constructor.
     * @param Category $categoryHelper
     * @param ResourceConnection $resource
     * @param null $name
     */
    public function __construct(
        Category $categoryHelper,
        ResourceConnection $resource,
        $name = null
    )
    {
        $this->categoryHelper = $categoryHelper;
        $this->resource = $resource;
        parent::__construct($name);
    }

    /**
     * Configure the command
     */
    protected function configure()
    {
        $this->setName('tools:regeneratecategory')
            ->setDescription('Regenerate Url\'s for categories')
            ->addOption(
                'cids', 'c',
                InputArgument::IS_ARRAY,
                'Categories to regenerate'
            )->addOption(
                'store', 's',
                InputArgument::IS_ARRAY,
                'Use the specific Store View'
            );
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     * @return int
     * @throws NoSuchEntityException
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $storesList = $this->getStoresToProcess($input->getOption('store'));
        $categoryIds = $this->getCategoriesToProcess($input->getOption('cids'));
        $progress  = new Progress();
        $progress->bind($output);
        $this->categoryHelper->regenerateUrlPathFromUrlKey(
            $progress,
            $categoryIds,
            $storesList
        );
        return 0;
    }

    /**
     * @param string $storeIds comma separate store ids
     * @return array
     */
    protected function getStoresToProcess($storeIds)
    {
        $allStores = $this->getAllStoreIds();

        if ($storeIds === null) {
            return $allStores;
        }

        return explode(',', $storeIds);
    }

    /**
     * @return array
     */
    public function getAllStoreIds()
    {
        $result = [];

        $sql = $this->resource->getConnection()->select()
            ->from($this->resource->getTableName('store'), ['store_id', 'code'])
            ->where('`code` <> ?', 'admin')
            ->order('store_id', 'ASC');

        foreach ($this->resource->getConnection()->fetchAll($sql) as $row) {
            if (isset($row['store_id']) && (int)$row['store_id'] > 0) {
                $result[(int)$row['store_id']] = $row['store_id'];
            }
        }

        return $result;
    }

    /**
     * @param $categoryIds
     * @return array|false|string[]
     */
    public function getCategoriesToProcess($categoryIds)
    {
        if ($categoryIds === null) {
            return [];
        }

        return explode(',', $categoryIds);;
    }
}
