<?php

namespace FiloBlu\ProductUrlTools\Helper;

use Exception;
use Magento\Catalog\Api\CategoryRepositoryInterface;
use Magento\Catalog\Api\Data\CategoryInterface;
use Magento\CatalogUrlRewrite\Model\CategoryUrlPathGenerator;
use Magento\CatalogUrlRewrite\Model\CategoryUrlRewriteGenerator;
use Magento\Eav\Api\AttributeRepositoryInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\DB\Adapter\AdapterInterface;
use Magento\Framework\EntityManager\MetadataPool;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Store\Model\ScopeInterface;
use Magento\UrlRewrite\Service\V1\Data\UrlRewrite;
use Zend_Db;
use Zend_Db_Expr;

use function count;

/**
 * Class Category
 * @package FiloBlu\ProductUrlTools\Helper
 */
class Category
{
    /**
     * @var string
     */
    protected $linkField;
    /**
     * @var ResourceConnection
     */
    protected $resourceConnection;
    /**
     * @var AttributeRepositoryInterface
     */
    protected $attributeRepository;
    /**
     * @var AdapterInterface
     */
    protected $connection;
    /**
     * @var MetadataPool
     */
    protected $metadataPool;
    /**
     * @var bool
     */
    protected $initialized;
    /**
     * @var array
     */
    protected $attributes = [];
    /**
     * @var ScopeConfigInterface
     */
    protected $scopeConfig;
    /**
     * @var string[]
     */
    protected $categoryUrlSuffix = [];
    /**
     * @var CategoryRepositoryInterface
     */
    protected $categoryRepository;

    /**
     * Category constructor.
     * @param ResourceConnection $resourceConnection
     * @param AttributeRepositoryInterface $attributeRepository
     * @param ScopeConfigInterface $scopeConfig
     * @param MetadataPool $metadataPool
     * @param CategoryRepositoryInterface $categoryRepository
     */
    public function __construct(
        ResourceConnection $resourceConnection,
        AttributeRepositoryInterface $attributeRepository,
        ScopeConfigInterface $scopeConfig,
        MetadataPool $metadataPool,
        CategoryRepositoryInterface $categoryRepository
    ) {
        $this->resourceConnection = $resourceConnection;
        $this->attributeRepository = $attributeRepository;
        $this->metadataPool = $metadataPool;
        $this->scopeConfig = $scopeConfig;
        $this->categoryRepository = $categoryRepository;
    }

    /**
     * @param $category
     * @param $storeId
     * @param $path
     * @throws Exception
     * @throws NoSuchEntityException
     */
    public function updateUrlPath($category, $storeId, $path)
    {
        $this->init();

        $table = $this->connection->getTableName('catalog_category_entity_varchar');
        $data = [
            $this->linkField => $category->getData($this->linkField),
            'attribute_id'   => $this->attributes['url_path']->getAttributeId(),
            'value'          => $path,
            'store_id'       => $storeId
        ];
        $this->connection->insertOnDuplicate($table, $data, ['value']);
    }

    /**
     * @throws NoSuchEntityException
     * @throws Exception
     */
    protected function init()
    {
        if ($this->initialized === true) {
            return;
        }

        $metadata = $this->metadataPool->getMetadata(CategoryInterface::class);
        $this->attributes['url_path'] = $this->attributeRepository->get($metadata->getEavEntityType(), 'url_path');
        $this->attributes['url_key'] = $this->attributeRepository->get($metadata->getEavEntityType(), 'url_key');
        $this->connection = $this->resourceConnection->getConnection();
        $this->linkField = $metadata->getLinkField();
        $this->initialized = true;
    }

    /**
     * @param Progress $progress
     * @throws NoSuchEntityException
     */
    public function regenerateUrlPathFromUrlKey(Progress $progress = null)
    {
        $this->init();

        $catalogCategoryEntityVarchar = $this->connection->getTableName('catalog_category_entity_varchar');
        $catalogCategoryEntity = $this->connection->getTableName('catalog_category_entity');
        $store = $this->connection->getTableName('store');
        $storeGroup = $this->connection->getTableName('store_group');
        $urlRewrite = $this->connection->getTableName('url_rewrite');

        $attributeId = $this->attributes['url_path']->getAttributeId();
        $lookup = [];

        $bind = [
            'categoryLevel' => CategoryUrlPathGenerator::MINIMAL_CATEGORY_LEVEL_FOR_PROCESSING - 1
        ];
        $linkField = $this->linkField;

        $query = $this->connection->select()
            ->from(['s' => $store], 'store_id')
            ->joinInner(['sg' => $storeGroup], 'sg.website_id  = s.website_id')
            ->joinLeft(['cce' => $catalogCategoryEntity],
                new Zend_Db_Expr("cce.path LIKE CONCAT( '1/', sg.root_category_id , '/%' )"))
            ->joinLeft(
                ['ccev' => $catalogCategoryEntityVarchar],
                "ccev.attribute_id = $attributeId AND ccev.$linkField = cce.$linkField AND ccev.store_id = s.store_id",
                [$linkField => "if (ccev.{$linkField}, ccev.{$linkField}, cce.{$linkField})", 'store_id' => 'if (ccev.store_id, ccev.store_id, s.store_id)', 'url_key' => 'value']
            )
            ->where('cce.level >= :categoryLevel')
            ->order('s.store_id', 'ASC');

        $categories = $this->connection->fetchAll($query, $bind, Zend_Db::FETCH_ASSOC);

        // Build lookup table
        foreach ($categories as $row) {
            $storeId = $row['store_id'];
            $entityId = $row['entity_id'];
            // TODO : category name <> urlkey
            $lookup[$storeId][$entityId] = $row['url_key'];
            if (isset($row['url_key'])) {
                if (isset($lookup[$storeId][$entityId]) &&  $lookup[$storeId][$entityId]) {
                    $row['url_key'] = $lookup[$storeId][$entityId];
                } else {
                    $cat = $this->categoryRepository->get($entityId, $storeId);
                    $row['url_key'] = $cat->formatUrlKey($cat->getName());
                }
            } else {
                $cat = $this->categoryRepository->get($entityId, $storeId);
                $row['url_key'] = $cat->formatUrlKey($cat->getName());
            }
            $lookup[$storeId][$entityId] = $row['url_key'];
        }

        $stores = $this->connection->fetchPairs( $this->connection->select()->from($store, ['store_id', 'code']));

        if ($progress) {
            foreach ($stores as $storeId  => $code) {
                if(!isset($lookup[$storeId])){
                    continue;
                }
                $progress->add($storeId, count($lookup[$storeId]), $code);
            }
        }

        if ($progress) {
            $progress->start();
        }
        // Process
        foreach ($categories as $row) {
            $storeId = $row['store_id'];
            $entityId = $row['entity_id'];

            if ($progress) {
                $progress->advance($storeId);
            }

            $path = explode('/', $row['path']);
            $newPath = [];

            foreach ($path as $slice) {
                if (!isset($lookup[$storeId][$slice])) {
                    continue;
                }

                if (strstr($lookup[$storeId][$slice], '/')) {
                    $urlKey = explode('/', $lookup[$storeId][$slice]);
                    $newPath[] = end($urlKey);
                } else {
                    $newPath[] = $lookup[$storeId][$slice];
                }
            }

            $urlPath = trim(implode('/', $newPath));

            $this->connection->insertOnDuplicate($catalogCategoryEntityVarchar, [
                $linkField     => $row[$linkField],
                'attribute_id' => $attributeId,
                'value'        => $urlPath,
                'store_id'     => $storeId
            ], ['value']);

            //Skip store zero on url_rewrite
            if ($storeId == 0) {
                continue;
            }

            //Update old url to autogenerated 0
            $updateCondition = [
                UrlRewrite::ENTITY_ID     => $entityId,
                UrlRewrite::REDIRECT_TYPE => 0,
                UrlRewrite::STORE_ID      => $storeId,
                UrlRewrite::ENTITY_TYPE   => CategoryUrlRewriteGenerator::ENTITY_TYPE
            ];

            $whereUpdateCondition = array_map(
                static function ($v, $k) {
                    return sprintf("%s='%s'", $k, $v);
                },
                $updateCondition,
                array_keys($updateCondition)
            );

            $this->connection->update(
                $urlRewrite,
                [UrlRewrite::IS_AUTOGENERATED => 0],
                implode(' AND ', $whereUpdateCondition)
            );

            //Insert Update url
            $this->connection->insertOnDuplicate(
                $urlRewrite,
                [
                    UrlRewrite::ENTITY_TYPE      => CategoryUrlRewriteGenerator::ENTITY_TYPE,
                    UrlRewrite::ENTITY_ID        => $entityId,
                    UrlRewrite::IS_AUTOGENERATED => 1,
                    UrlRewrite::REDIRECT_TYPE    => 0,
                    UrlRewrite::REQUEST_PATH     => $urlPath . $this->getCategoryUrlSuffix($storeId),
                    UrlRewrite::STORE_ID         => $storeId,
                    UrlRewrite::TARGET_PATH      => "catalog/category/view/id/$entityId"
                ],
                [UrlRewrite::IS_AUTOGENERATED, UrlRewrite::REDIRECT_TYPE, UrlRewrite::TARGET_PATH]
            );
        }
    }

    /**
     * Retrieve category rewrite suffix for store
     *
     * @param int $storeId
     * @return string
     */
    protected function getCategoryUrlSuffix($storeId)
    {
        if (!isset($this->categoryUrlSuffix[$storeId])) {
            $this->categoryUrlSuffix[$storeId] = $this->scopeConfig->getValue(
                CategoryUrlPathGenerator::XML_PATH_CATEGORY_URL_SUFFIX,
                ScopeInterface::SCOPE_STORE,
                $storeId
            );
        }
        return $this->categoryUrlSuffix[$storeId];
    }
}
