<?php


namespace FiloBlu\ProductUrlTools\Console\Command;

use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Helper\ProgressBar;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\ResourceConnection;
use Psr\Log\LoggerInterface;
use Magento\Catalog\Model\ResourceModel\Eav\Attribute;
use Magento\Eav\Model\Entity\Attribute\Option;
use FiloBlu\ProductUrlTools\Model\ProductFactory;
use Magento\Framework\DB\TransactionFactory;
use Symfony\Component\Console\Style\SymfonyStyle;
use FiloBlu\ProductUrlTools\Helper\Data;

class CheckDuplicateAttributes extends Command
{

    const INPUT_ATTRIBUTE_NAME = 'attribute_name';
    const INPUT_COLUMNS = 'columnns';

    /**
     * @var ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * @var \Magento\Framework\DB\Adapter\AdapterInterface
     */
    protected $resourceConnection;

    /**
     * @var LoggerInterface
     */
    protected $_logger;

    /**
     * @var Attribute
     */
    protected $eavAttribute;

    /**
     * @var ProductFactory
     */
    protected $productFactory;

    /**
     * @var TransactionFactory
     */
    protected $transactionFactory;

    /**
     * @var Data
     */
    protected $helper;

    /**
     * @var Option
     */
    protected $option;

    public function __construct(
        ScopeConfigInterface $scopeConfig,
        ResourceConnection $resourceConnection,
        LoggerInterface $logger,
        Attribute $eavAttribute,
        ProductFactory $productFactory,
        TransactionFactory $transactionFactory,
        Data $helper,
        Option $option,
        $name = null
    )
    {
        $this->scopeConfig = $scopeConfig;
        $this->resourceConnection = $resourceConnection->getConnection();
        $this->_logger = $logger;
        $this->eavAttribute = $eavAttribute;
        $this->productFactory = $productFactory;
        $this->transactionFactory = $transactionFactory;
        $this->helper = $helper;
        $this->option = $option;
        parent::__construct($name);
    }

    protected function configure()
    {

        $this->setName('tools:duplicatedattributes:find')
            ->setDescription('Check for duplicated attributes and store their products into a temp table "filoblu_duplicated_attribute_option"')
            ->addArgument(
                self::INPUT_ATTRIBUTE_NAME,
                InputArgument::OPTIONAL,
                'Attribute Code'
            )
            ->addArgument(
                self::INPUT_COLUMNS,
                InputArgument::IS_ARRAY | InputArgument::OPTIONAL,
                'Columns'
            );

        parent::configure();

    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $io = new SymfonyStyle($input, $output);
        $io->write('');
        $io->title('FiloBlu Duplicate Attribute Options Checker');

        $adminConfig = $this->scopeConfig->getValue('commandtools/general/attribute_name', \Magento\Store\Model\ScopeInterface::SCOPE_STORE);
        $limit = (int)$this->scopeConfig->getValue('commandtools/general/items_limit', \Magento\Store\Model\ScopeInterface::SCOPE_STORE);
        if(!$limit){
            $limit = 1000;
        }

        //get attribute name as input otherwise leave the default setted inside admin section
        $value = $adminConfig;
        if (!$adminConfig) {
            $value = strtolower($input->getArgument(self::INPUT_ATTRIBUTE_NAME));
        }

        //get optional columns to show
        if ($input->getArgument(self::INPUT_COLUMNS)) {
            $columns = implode(", ", $input->getArgument(self::INPUT_COLUMNS));
        }

        $attributes = $this->eavAttribute->getCollection()->addFieldToFilter('attribute_code', ['eq' => $value]);
        $backendType = 'int';

        if (!$attributes->getData()) {
            $types = $this->helper->getAttributeTypes();

            if ($value) {
                $io->error("Cannot find any attribute named '{$value}'");
            } else {
                $io->error("Attribute code cannot be null");
            }


            $attributeTypes = [];


            foreach ($types as $type) {
                $attributeTypes[] = $type['attribute_code'];
            }

            $value = $io->choice('Avalilable attributes are', $attributeTypes);

            //get backend_type
            $attributes = $this->eavAttribute->getCollection()->addFieldToFilter('attribute_code', ['eq' => $value]);
            $attr = $attributes->getData();

            $backendType = $attr[0]['backend_type'];

            $io->success($value . " attribute selected!");
            $io->writeln('');


        }

        //get all attribute options with a particulare attribute_code
        $optionsQuery = $this->resourceConnection->select()
            ->from('eav_attribute_option_value', ['value_id', 'value'])
            ->join('eav_attribute_option', 'eav_attribute_option.option_id = eav_attribute_option_value.option_id', ['option_id'])
            ->join('eav_attribute', 'eav_attribute_option.attribute_id = eav_attribute.attribute_id')
            ->where('eav_attribute_option_value.store_id = 0')
            ->where("eav_attribute.attribute_code = '{$value}'")
            ->order('eav_attribute_option_value.value ASC');

        $collection = $this->resourceConnection->fetchAll($optionsQuery);
        $colors = [];

        //loop inside data taken from db
        foreach ($collection as $item) {
            foreach ($item as $key => $val) {
                //if (isset($item['frontend_input']) && $item['frontend_input'] == 'select') {
                    $colors[strtolower(trim($item['value']))][$item['option_id']] = $item['option_id'];
                //}
            }
        }

        $unassociatedOptions = [];

        foreach ($colors as $color => $ids) {

            $idToUse = min($ids);

            $collection = $this->productFactory->create();

            if (count($ids) > 1) {

                foreach ($ids as $id) {
                    if ($id != $idToUse) {
                        //$output->writeln($id);
                        $products = $this->helper->getProducts($id, $backendType);

                        if (!$products) {
                            $unassociatedOptions[$id] = $id;
                            continue;
                        }

                        $io->table(
                            [
                                'Origial ID',
                                'Attribute',
                                'Duplicated ID',
                                'Products found'
                            ],
                            [
                                [$idToUse, $color, $id, number_format(count($products), 0, "", ".")]
                            ]
                        );
                        $format = "d/m/Y H:i:s";
                        $start = time();
                        $output->writeln(" Started at: <info>".date($format, $start)."</info> ");



                        $progressBar = new ProgressBar($output);

                        $progressBar->start(count($products));

                        $step = 0;
                        $count = 0;

                        $columns = [
                            'entity_id',
                            'sku',
                            'attribute_id',
                            'original_option_id',
                            'attribute_code',
                            'attribute_value',
                            'backend_type',
                            'option_id',
                            'processed'
                        ];

                        $insertQuery = '';

                        $commandToolsProducts = $collection->getCollection();


                        foreach ($products as $key => $product) {

                            if ($count == 0) {
                                $processed = false;
                            }

                            $commandToolsProducts->addFieldToFilter('sku', $product['sku'])
                            ->addFieldToFilter('option_id', $id);

                            if ($commandToolsProducts->getData()) {
                                continue;
                            }

                            $values = [
                                $product['entity_id'],
                                "'" . $product['sku'] . "'",
                                $product['attribute_id'],
                                $idToUse,
                                "'" . $value . "'",
                                "'" . $product['value'] . "'",
                                "'" . $backendType . "'",
                                $product['option_id'],
                                0
                            ];


                            if ($count == 0) {
                                $output->write(" | ");
                                $insertQuery .= "INSERT INTO `filoblu_duplicated_attribute_option` (" . implode(", ",  $columns) . ") VALUES (" . implode(", ", $values)  . ")";
                            } else {
                                $insertQuery .= ", (" . implode(", ", $values)  . ")";
                            }

                            $progressBar->advance();

                            if (($count + 1) % $limit === 0) {

                                $step += $limit;

                                try {

                                    $this->resourceConnection->beginTransaction();
                                    $this->resourceConnection->query($insertQuery);
                                    $this->resourceConnection->commit();
                                    sleep(1);
                                } catch (\Exception $e) {
                                    $output->writeln($e->getMessage());
                                    $this->resourceConnection->rollBack();
                                }


                                $insertQuery = '';
                                $processed = true;
                                $count = -1;
                            } else {
                                $processed = false;
                            }

                            $count++;
                        }

                        if (!$processed && $insertQuery) {
                            $this->resourceConnection->query($insertQuery);
                            $insertQuery = '';
                        }
                        $progressBar->finish();


                        $end = time();
                        $time = date('H:i:s', ($end - $start));
                        $io->writeln("\n Ended at: <info>". date($format, $end) . "</info>");
                        $io->writeln("\n Time elapsed: <info>{$time}</info>");
                        $io->newLine();
                        $io->writeln('Row correctly stored!');
                        $io->newLine();
                    }
                }
            }
        }


        if ($unassociatedOptions) {

            $toDelete = [];

            foreach ($unassociatedOptions as $optionId) {

                $option = $this->helper->getAttributeOptionValueById($optionId);

                if ($option[0]['value']) {
                    $toDelete[] = [
                        'option_id' => $option[0]['option_id'],
                        'value' => $option[0]['value']
                    ];
                }
            }


            if ($toDelete) {
                $io->note("Following attribute has no products associated and can be deleted:");
                $io->table(
                    [
                        'Option ID',
                        'Value'
                    ],
                    $toDelete
                );

                $question = "Would you like do delete it ? (yes|no)";

                if (count($toDelete) > 1) {
                    $question = "Would you like do delete them ? (yes|no)";
                }

                $io->ask($question, 'no', function($answer) use ($output, $toDelete) {

                    if (strtolower($answer[0]) == 'y') {
                        //$output->writeln($answer);
                        foreach ($toDelete as $key => $item) {
                            $toDelete[$key]['delete'][] = $item['option_id'];
                            if (isset($toDelete[$key]['delete'])) {
                                if (count($toDelete[$key]['delete']) >= 1) {
                                    // DELETE DUPLICATE ATTRIBUTE OPTIONS WITH NO PRODUCTS ASSOCIATED
                                    foreach ($toDelete[$key]['delete'] as $optionId) {
                                        $optionModel = $this->option->load($optionId);
                                        try {
                                            $optionModel->delete();
                                            $output->writeln("<info>{$key} ({$optionId}) Option Deleted!</info>");
                                            //echo '<font color="green">"'.$key.' ('.$optionId.')" Option Deleted!</font><br />';
                                        }
                                        catch(Exception $e) {
                                            $output->writeln("<error>{$e->getMessage()}</error>");
                                            //echo '<font color="red">'. $e->getMessage() .'</font><br />';
                                        }
                                    }
                                }
                            }
                        }
                    }
                });

            }
        }
      // exit;
    }
}