<?php

namespace FiloBlu\ProductUrlTools\Controller\Adminhtml\Product;

use Exception;
use FiloBlu\ProductUrlTools\Console\Command\RegenerateProductUrlKeyCommand;
use FiloBlu\ProductUrlTools\Helper\Data;
use Magento\Backend\App\Action\Context;
use Magento\Backend\Model\View\Result\RedirectFactory;
use Magento\Catalog\Controller\Adminhtml\Product;
use Magento\Catalog\Controller\Adminhtml\Product\Builder;
use Magento\Catalog\Model\ResourceModel\Product\ActionFactory;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory;
use Magento\CatalogUrlRewrite\Model\ProductUrlRewriteGenerator;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Message\ManagerInterface;
use Magento\Ui\Component\MassAction\Filter;
use Magento\UrlRewrite\Model\UrlPersistInterface;
use Magento\UrlRewrite\Service\V1\Data\UrlRewrite;
use Psr\Log\LoggerInterface;
use Magento\Catalog\Model\ProductFactory;
use Magento\Framework\EntityManager\MetadataPool;
use Magento\Catalog\Api\Data\ProductInterface;

/**
 *
 */
class RegenerateUrl extends Product
{
    /**
     * @var Filter
     */
    protected $_filter;
    /**
     * @var CollectionFactory
     */
    protected $_collectionFactory;
    /**
     * @var LoggerInterface
     */
    protected $_loggerInterface;
    /**
     * @param Context
     */
    protected $_context;
    /**
     * @var ManagerInterface
     */
    protected $_messageManager;
    /**
     * @var RedirectFactory
     */
    protected $_resultRedirectFactory;
    /**
     * @var ResourceConnection
     */
    protected $resource;
    /**
     * @var Data
     */
    public $helperData;
    /**
     * @var RegenerateProductUrlKeyCommand
     */
    public $regenerateUrlKey;
    /**
     * @var UrlPersistInterface
     */
    private $urlPersist;
    /**
     * @var ActionFactory
     */
    private $actionFactory;
    /**
     * @var ProductUrlRewriteGenerator
     */
    private $productUrlRewriteGenerator;
    /**
     * @var ProductFactory
     */
    private $productFactory;

    /**
     * @var \Magento\Framework\EntityManager\MetadataPool
     */
    private $metadataPool;


    /**
     * @param Context $_context
     * @param Builder $_productBuilder
     * @param Filter $_filter
     * @param CollectionFactory $_collectionFactory
     * @param ResourceConnection $resource
     * @param LoggerInterface $logger
     * @param Data $helperData
     * @param RegenerateProductUrlKeyCommand $regenerateUrlKey
     * @param ActionFactory $actionFactory
     * @param ProductUrlRewriteGenerator $productUrlRewriteGenerator
     * @param UrlPersistInterface $urlPersist
     * @param ProductFactory $productFactory
     * @param MetadataPool $metadataPool
     */
    public function __construct(
        Context                        $_context,
        Builder                        $_productBuilder,
        Filter                         $_filter,
        CollectionFactory              $_collectionFactory,
        ResourceConnection             $resource,
        LoggerInterface                $logger,
        Data                           $helperData,
        RegenerateProductUrlKeyCommand $regenerateUrlKey,
        ActionFactory                  $actionFactory,
        ProductUrlRewriteGenerator     $productUrlRewriteGenerator,
        UrlPersistInterface            $urlPersist,
        ProductFactory $productFactory,
        MetadataPool $metadataPool
    ) {
        $this->_collectionFactory = $_collectionFactory;
        $this->_filter = $_filter;
        $this->_loggerInterface = $logger;
        $this->_context = $_context;
        $this->resource = $resource;
        $this->_messageManager = $this->_context->getMessageManager();
        $this->_resultRedirectFactory = $this->_context->getResultRedirectFactory();
        $this->helperData = $helperData;
        $this->regenerateUrlKey = $regenerateUrlKey;
        $this->actionFactory = $actionFactory;
        $this->productUrlRewriteGenerator = $productUrlRewriteGenerator;
        $this->urlPersist = $urlPersist;
        parent::__construct($this->_context, $_productBuilder);
        $this->productFactory = $productFactory;
        $this->metadataPool = $metadataPool;
    }

    /**
     * Execute action based on request and return result
     *
     * @return \Magento\Backend\Model\View\Result\Redirect|\Magento\Framework\Controller\Result\Redirect
     * @throws \Magento\Framework\Exception\NotFoundException|\Magento\Framework\Exception\LocalizedException
     */
    public function execute()
    {
        $this->_loggerInterface->debug('---------------------START URL ACTION---------------------');

        $collection = $this->_filter->getCollection($this->_collectionFactory->create());
        $collection->addAttributeToSelect('entity_id', 'row_id');
        $productIds = $collection->getAllIds();
        $configData = $this->helperData->getAttributeConfig();

        $collection->addAttributeToSelect(array_merge(['name', 'url_key'], $configData));
        if (0 !== count($productIds)) {
            if (count($productIds) <= 200) {
                if (isset(
                    $this->getRequest()
                         ->getParam('filters')['store_id']
                )) {
                    $storeIds = [$this->getRequest()->getParam('filters')['store_id']];
                } else {
                    $storeIds = $this->regenerateUrlKey->getAllStoreIds();
                }
                try {
                    foreach ($storeIds as $storeId) {
                        $collection->setStore($storeId);
                        foreach ($productIds as $product) {
                            $this->processUrlKey($product, $configData, $storeId, $collection);
                            $this->processUrlPath($storeId, $product);
                        }
                    }
                    $this->_messageManager->addSuccessMessage(__('%1 product(s) have been update giovanotto/a', count($productIds)));
                    $this->_loggerInterface->info(count($productIds) . ' product(s) have been update.' . "\nPRODUCT(S) => " . json_encode($productIds));
                } catch (LocalizedException $e) {
                    $this->_loggerInterface->error($e);
                    $this->_messageManager->addError($e->getMessage());
                } catch (Exception $e) {
                    $this->_loggerInterface->error($e);
                    $this->_getSession()->addException($e, __('Something went wrong while updating the product(s).'));
                }
            } else {
                $this->_messageManager->addErrorMessage(__('select maximum 200 products'));
            }
        } else {
            $this->_messageManager->addErrorMessage(__('No Products Selected'));
        }

        $this->_loggerInterface->debug('---------------------END URL ACTION---------------------');

        return $this->_resultRedirectFactory->create()
                                            ->setPath($this->getUrl('catalog/product/index'));
    }

    /**
     * @param $product
     * @param $configData
     * @param $storeId
     * @param $collection
     * @return void
     * @throws \Exception
     */
    public function processUrlKey($product, $configData, $storeId, $collection)
    {
        $productObj = $this->productFactory->create();
        $productObj->setStoreId($storeId);
        $productObj->load($product);
        $attributes = $productObj->getData();

        $filteredAttributes = [];
        foreach ($configData as $attributeName) {
            if (isset($attributes[$attributeName]) && $attributes[$attributeName]) {
                $filteredAttributes[$attributeName] = $attributes[$attributeName];
            }
            if ($attributeName === 'store_code') {
                $filteredAttributes['store_code'] = $this->helperData->getStoreCode($storeId);
            }
        }

        foreach ($filteredAttributes as $key => &$attributeName) {
            if ($attributeName === 'entity_id') {
                $attributeName = $productObj->getId();
            }
        }

        $name = $productObj->getName();
        if (in_array($name, $filteredAttributes)) {
            unset($filteredAttributes['name']);
        }
        $attribute = str_replace(' ', '-', strtolower(implode('-', $filteredAttributes)));
        $urlKey = $productObj->formatUrlKey($name . '-' . $attribute);
        $attributesData = ['url_key' => $urlKey];
        $action = $this->actionFactory->create();
        // Update Attribute urlkey
        $action->updateAttributes([$productObj->getId()], $attributesData, $storeId);
    }

    /**
     * @param $storeId
     * @param $product
     * @return void
     * @throws \Magento\UrlRewrite\Model\Exception\UrlAlreadyExistsException
     */
    public function processUrlPath($storeId, $product)
    {
        $linkField = $this->metadataPool->getMetadata(ProductInterface::class)->getLinkField();
        $productObj = $this->productFactory->create();
        $productObj->setStoreId($storeId);
        $productObj->load($product);

        //delete old redirect
        $this->regenerateUrlKey->deleteRedirect($storeId, $productObj->getData($linkField));
        $this->urlPersist->deleteByData([
            UrlRewrite::ENTITY_ID => $productObj->getData($linkField),
            UrlRewrite::ENTITY_TYPE => ProductUrlRewriteGenerator::ENTITY_TYPE,
            UrlRewrite::REDIRECT_TYPE => 0,
            UrlRewrite::STORE_ID => $storeId
        ]);
        //generate new rewrite
        $urls = $this->productUrlRewriteGenerator->generate($productObj);
        $this->urlPersist->replace($urls);
    }
}
