<?php


namespace FiloBlu\ProductUrlTools\Console\Command;

use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\ResourceConnection;
use Psr\Log\LoggerInterface;
use Magento\Catalog\Model\ResourceModel\Eav\Attribute;
use FiloBlu\ProductUrlTools\Model\ProductFactory;
use Magento\Framework\DB\TransactionFactory;
use Symfony\Component\Console\Question\Question;
use Symfony\Component\Console\Style\SymfonyStyle;
use FiloBlu\ProductUrlTools\Model\ResourceModel\Product\CollectionFactory as ProductCollectionFactory;
use FiloBlu\ProductUrlTools\Helper\Data;
use Magento\Eav\Model\ResourceModel\Entity\Attribute\Option\Collection as OptionCollection;
use Magento\Eav\Model\ResourceModel\Entity\Attribute\Option\CollectionFactory as OptionCollectionFactory;
use Magento\Eav\Model\Entity\Attribute\Option;


class RemoveDuplicateAttributes extends Command
{

    const INPUT_ELEMENTS_TO_PROCESS = 'elements_to_process';

    /**
     * @var ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * @var \Magento\Framework\DB\Adapter\AdapterInterface
     */
    protected $resourceConnection;

    /**
     * @var LoggerInterface
     */
    protected $_logger;

    /**
     * @var Attribute
     */
    protected $eavAttribute;

    /**
     * @var ProductFactory
     */
    protected $productFactory;

    /**
     * @var TransactionFactory
     */
    protected $transactionFactory;

    /**
     * @var Data
     */
    protected $helper;


    /**
     * @var ProductCollectionFactory
     */
    protected $productCollectionFactory;

    /**
     * @var OptionCollection
     */
    protected $optionCollection;


    /**
     * @var OptionCollectionFactory
     */
    protected $optionCollectionFactory;

    /**
     * @var Option
     */
    protected $option;


    public function __construct(
        ScopeConfigInterface $scopeConfig,
        ResourceConnection $resourceConnection,
        LoggerInterface $logger,
        Attribute $eavAttribute,
        ProductFactory $productFactory,
        TransactionFactory $transactionFactory,
        ProductCollectionFactory $productCollectionFactory,
        Data $helper,
        OptionCollection $optionCollection,
        OptionCollectionFactory $optionCollectionFactory,
        Option $option,
        $name = null
    )
    {
        $this->scopeConfig = $scopeConfig;
        $this->resourceConnection = $resourceConnection->getConnection();
        $this->_logger = $logger;
        $this->eavAttribute = $eavAttribute;
        $this->productFactory = $productFactory;
        $this->transactionFactory = $transactionFactory;
        $this->productCollectionFactory = $productCollectionFactory;
        $this->helper = $helper;
        $this->optionCollection = $optionCollection;
        $this->optionCollectionFactory = $optionCollectionFactory;
        $this->option = $option;
        parent::__construct($name);
    }

    protected function configure()
    {
        $this->setName('tools:duplicatedattributes:delete')
            ->setDescription('Remove duplicated options and associate their products to the original attribute picking info from "filoblu_duplicated_attribute_option" table')
            ->addArgument(
                self::INPUT_ELEMENTS_TO_PROCESS,
                InputArgument::OPTIONAL,
                'N° of elements to process'
            );
        parent::configure();

    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $numberOfItems = 100;
        if ($input->getArgument(self::INPUT_ELEMENTS_TO_PROCESS)) {
            $numberOfItems = $input->getArgument(self::INPUT_ELEMENTS_TO_PROCESS);
        }

        $io = new SymfonyStyle($input, $output);
        $io->write('');
        $io->title('FiloBlu Duplicate Attribute Options Checker - Product Process');


        $productsCollectionFactory = $this->productFactory->create()->getCollection();

        $productsCollectionFactory
            ->addFieldToSelect(['attribute_value', 'original_option_id', 'option_id', 'attribute_code'])
            ->addFieldToFilter('processed', ['eq' => 0]);

        $productsCollectionFactory->getSelect()->group('attribute_value');

        $productsCollectionFactoryData = $productsCollectionFactory->getData();

        if (!$productsCollectionFactoryData) {
            $io->error('Nothing to process');
            exit;
        }

        $list = [];

        foreach ($productsCollectionFactoryData as $item) {
            $list[] = trim($item['attribute_value']);
        }


        $io->section('Available attributes list:');
        $io->listing($list);


        $helper = $this->getHelper('question');


        $question = new Question('<info>Which attribute would you like to select ? (Type "all" to get all items and "@" to exit):</info> ', 'All');
        $question->setAutocompleterValues(array_merge($list, ['All']));
        $bundleName = '';

        $attributsToMerge = [];

        while ($bundleName != '@') {
            $bundleName = $helper->ask($input, $output, $question);

            if ($bundleName != '' && $bundleName != '@') {
                if (strtolower($bundleName) == 'all' && count($attributsToMerge)) {
                    break;
                }

                if (strtolower($bundleName) == 'all') {
                    $attributsToMerge = $list;
                    break;
                } else {
                    $attributsToMerge[] = $bundleName;
                }

            }
        }


        if (!$attributsToMerge) {
            $io->error("Nothing was selected, exit!");
            return false;
        }
        $io->writeln("");

        foreach ($attributsToMerge as $item) {

            $optionsProductsCount = [];

            $key = $this->helper->arraySearch($item, $productsCollectionFactoryData, 'attribute_value');
            $originalOptionId = $productsCollectionFactoryData[$key]['original_option_id'];
            $attributeCode = $productsCollectionFactoryData[$key]['attribute_code'];

            //check how many products this option is associated with
            $originalProductsCount = count($this->helper->getProducts($originalOptionId));

            $io->success("Option '{$item}' selected with option_id: {$originalOptionId} - Products count: {$originalProductsCount}");

            $optionsProductsCount[] = [
                'option_id' => $originalOptionId,
                'products' => $originalProductsCount
            ];

            $io->writeln("");


            $duplicatedOptionItems = $this->helper->getAttributeOptionValueByValue($item, $originalOptionId);

            foreach ($duplicatedOptionItems as $duplicated) {
                $duplicatedOptionId = $duplicated['option_id'];
                $duplicatedOptionData = $this->helper->getAttributeOptionValueById($duplicatedOptionId);
                $duplicatedProductsCount = count($this->helper->getProducts($duplicatedOptionId));

                $optionsProductsCount[] = [
                    'option_id' => $duplicatedOptionId,
                    'products' => $duplicatedProductsCount
                ];

                $io->section("Duplicated Option ID: {$duplicatedOptionId}");
                $output->writeln("Duplicate products count: {$duplicatedProductsCount}");
                $output->writeln("Original option: {$originalOptionId} - duplicated option: {$duplicatedOptionId}");
                $io->writeln("");
            }


            //sort array by products count in order to get the option_id with highest products count
            usort($optionsProductsCount, [$this->helper, "sortCount"]);


            $io->comment("Checking options products count...");

            $newOptionId = $optionsProductsCount[0]['option_id'];

            $io->comment("Original option_id has changed to {$newOptionId}");

            $productsCollectionFactory = $this->productFactory->create()->getCollection();


            foreach ($optionsProductsCount as $key => $option) {
                if ($option['option_id'] == $newOptionId) {
                    continue;
                }

                $io->comment("Products associated to option_id {$option['option_id']} will be associated to option_id {$newOptionId}");

                $productsCollectionFactory->addFieldToFilter(
                        'original_option_id',
                        ['eq' => $originalOptionId]
                    )
                    ->addFieldToFilter(
                        'processed',
                        ['eq' => false]
                    );



                //get products row id from custom table
                $customProducts = $productsCollectionFactory->getData();

                if (!count($customProducts)) {
                    continue;
                }

                $productIds = [];

                foreach ($customProducts as $customProduct) {
                    $productIds[] = $customProduct['entity_id'];
                }

                $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
                $productActionObject = $objectManager->create('Magento\Catalog\Model\Product\Action');
                $productActionObject->updateAttributes($productIds, array($attributeCode => $newOptionId), 0);

                $productsCollectionFactory = $this->productFactory->create()->getCollection();

                //updating processed column value for the current attribute
                foreach ($customProducts as $customProduct) {

                    $prod = $this->productFactory->create()->load($customProduct['id']);
                    $prod->setProcessed(true)->save();
                }

                $io->text("Deleting empty option value with id {$option['option_id']}...");

                $optionModel = $this->option->load($option['option_id']);
                try {
                    $optionModel->delete();
                    $output->writeln(" <info>{$key} ({$option['option_id']}) Option Deleted!</info>");
                    //echo '<font color="green">"'.$key.' ('.$optionId.')" Option Deleted!</font><br />';
                }
                catch(Exception $e) {
                    $output->writeln(" <error>{$e->getMessage()}</error>");
                    //echo '<font color="red">'. $e->getMessage() .'</font><br />';
                }
            }
        }
    }


    private function compareOptions($arr1, $arr2)
    {
        $output = [];

        switch (true) {
            case (count($arr1) == count($arr2)):
                foreach ($arr1 as $key => $item) {
                    $id = $this->helper->arraySearch($item, $arr2, 'store_id');
                    if (trim($item['value']) == trim($arr2[$key]['value'])) {
                        $output[$key] = $item;
                        $output[$key]['value'] = $arr2[$id]['value'];
                    }
                }
                break;
            case (count($arr1) > count($arr2) && count($arr2) > 1):
                foreach ($arr1 as $key => $item) {
                    if (isset($arr2[$key])) {

                    }
                }
                break;
            case (count($arr1) > count($arr2) && count($arr2) == 1):
                return $arr1;
                break;
            default:
                break;

        }

        return $output;
    }
}