<?php

namespace FiloBlu\ProductUrlTools\Console\Command;

use Exception;
use FiloBlu\ProductUrlTools\Helper\Data;
use Magento\Catalog\Model\Product\Attribute\Source\Status;
use Magento\Catalog\Model\Product\Visibility;
use Magento\Catalog\Model\ResourceModel\Product\Action;
use Magento\Catalog\Model\ResourceModel\Product\ActionFactory;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory;
use Magento\CatalogUrlRewrite\Model\ProductUrlRewriteGenerator;
use Magento\Framework\App\ResourceConnection;
use Magento\UrlRewrite\Model\UrlPersistInterface;
use Magento\UrlRewrite\Model\UrlRewriteFactory;
use Magento\UrlRewrite\Service\V1\Data\UrlRewrite;
use Psr\Log\LoggerInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Question\Question;
use Symfony\Component\Console\Style\SymfonyStyle;

use function sprintf;

/**
 * Class RegenerateProductUrlKeyCommand
 * @package FiloBlu\ProductUrlTools\Console\Command
 */
class RegenerateProductUrlKeyCommand extends Command
{
    /**
     * @var CollectionFactory
     */
    protected $collectionFactory;

    /**
     * @var LoggerInterface
     */
    protected $logger;

    /**
     * @var Action
     */
    protected $action;

    /**
     * @var ResourceConnection
     */
    protected $resource;

    /**
     * @var Status
     */
    protected $productStatus;

    /**
     * @var Visibility
     */
    protected $productVisibility;

    /**
     * @var ActionFactory
     */
    protected $actionFactory;

    /**
     * @var ExecutorGroup
     */
    protected $executorGroup;

    /**
     * @var UrlRewriteFactory
     */
    protected $urlRewriteFactory;

    /**
     * @var SymfonyStyle $io
     */
    protected $io;

    /**
     * @var UrlPersistInterface
     */
    protected $urlPersist;

    /**
     * @var ProductUrlRewriteGenerator
     */
    protected $productUrlRewriteGenerator;

    /**
     * @var array
     */
    private $oldUrls;

    /**
     * @var array
     */
    private $urlKeys;

    /**
     * @var false
     */
    private $generate301RedirectConfig;

    /**
     * @var Data
     */
    private $helperData;

    /**
     * RegenerateProductUrlKeyCommand constructor.
     * @param CollectionFactory $collectionFactory
     * @param LoggerInterface $logger
     * @param Status $productStatus
     * @param Visibility $productVisibility
     * @param ResourceConnection $resource
     * @param ActionFactory $actionFactory
     * @param ExecutorGroup $executorGroup
     * @param UrlRewriteFactory $urlRewriteFactory
     * @param \Magento\UrlRewrite\Model\UrlPersistInterface $urlPersist
     * @param \Magento\CatalogUrlRewrite\Model\ProductUrlRewriteGenerator $productUrlRewriteGenerator
     * @param Data $helperData
     */
    public function __construct(
        CollectionFactory $collectionFactory,
        LoggerInterface $logger,
        Status $productStatus,
        Visibility $productVisibility,
        ResourceConnection $resource,
        ActionFactory $actionFactory,
        ExecutorGroup $executorGroup,
        UrlRewriteFactory $urlRewriteFactory,
        UrlPersistInterface $urlPersist,
        ProductUrlRewriteGenerator $productUrlRewriteGenerator,
        Data $helperData
    ) {
        $this->collectionFactory = $collectionFactory;
        $this->logger = $logger;
        $this->productVisibility = $productVisibility;
        $this->productStatus = $productStatus;
        $this->actionFactory = $actionFactory;
        $this->resource = $resource;
        $this->executorGroup = $executorGroup;
        $this->urlRewriteFactory = $urlRewriteFactory;
        $this->urlPersist = $urlPersist;
        $this->productUrlRewriteGenerator = $productUrlRewriteGenerator;
        $this->oldUrls = [];
        $this->urlKeys = [];
        $this->generate301RedirectConfig = false;
        $this->helperData = $helperData;

        parent::__construct();
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     * @return int|void|null
     */
    public function execute(InputInterface $input, OutputInterface $output)
    {
        $this->logger->debug('---------------------START URLKEY COMMAND---------------------');

        $storesList = $this->getStoresToProcess($input->getOption('store'));
        $productIds = $input->getArgument('pids');
        $allVisible = $input->getOption('all_visible');
        $from = $input->getOption('from');
        $jobs = $input->getOption('jobs');
        $this->action = $this->actionFactory->create();

        $info = '
 _____     _    _____         _
|  |  |___| |  |_   _|___ ___| |___
|  |  |  _| |    | | | . | . | |_ -|
|_____|_| |_|    |_| |___|___|_|___|
                                    ';
        $output->writeln($info);

        if (isset($productIds[0]) && $productIds[0] === 'pids') {
            unset($productIds[0]);
            $productIds = array_values($productIds);
        }

        $helper = $this->getHelper('question');

        $this->io = new SymfonyStyle($input, $output);

        $userDefaultRewriteConfig = true;
        $attributeName = '';
        $attributesNames = [];

        $useDefaultConfig = $this->io->ask(
            'Do you want to use the default url configuration consisting on product_name-store_id-product_id  ? (Yes|No)',
            'Yes'
        );

        if (strtolower(trim($useDefaultConfig[0])) === 'n' && !preg_match('/\W.*/im', $useDefaultConfig)) {
            $userDefaultRewriteConfig = false;

            $this->io->success(
                'You have chosen not to use the default configuration, the variant consists of product_name--selected_attribute_value'
            );

            $attributesName = $this->getAttributesName();
            $question = new Question('Please, type the attribute name to use (type @ to exit) : ', 'entity_id');
            $question->setAutocompleterValues($attributesName);

            do {
                $attributeName = $helper->ask($input, $output, $question);

                if ($attributeName !== '@') {
                    $attributesNames[] = $attributeName;
                }
            } while ($attributeName !== '@');

            $generate301Redirect = $this->io->ask('Do you want to allow system to generate 301 rediects for the new urls ? (Yes|No)', 'No');

            if (strtolower(trim($generate301Redirect[0])) == 'y' && !preg_match("/\W.*/im", $generate301Redirect)) {
                $this->generate301RedirectConfig = true;
            }
        }

        $loggerData = [
            'store_list' => implode(',', $storesList),
            'product_ids_count' => count($productIds),
            'user_default_rewrite_config' => $userDefaultRewriteConfig,
            'attributes_name' => implode(',', $attributesNames),
            'all_visible' => $allVisible,
            'from' => $from,
            'jobs' => $jobs,
            '301_redirects' => $this->generate301RedirectConfig
        ];
        $this->logger->info("Input data: " . print_r($loggerData, true));
        unset($loggerData);

        if ($jobs <= 1) {
            $section = null;

            $this->logger->debug('START URLKEY GENERATOR');
            //Regenerate url Keys
            foreach ($storesList as $storeId) {
                $this->regenerateUrlKey(
                    $storeId,
                    $allVisible,
                    $productIds,
                    $from,
                    $section,
                    $userDefaultRewriteConfig,
                    $attributesNames
                );
            }

            //Regenerate urls
            if (!$userDefaultRewriteConfig) {
                $this->logger->debug('START URL GENERATOR');
                foreach ($storesList as $storeId) {
                    $this->regenerateUrls(
                        $storeId,
                        $allVisible,
                        $productIds,
                        $from,
                        $section,
                        $userDefaultRewriteConfig,
                        $attributesNames
                    );
                }
            }

            $this->logger->debug('-------------------------END COMMAND--------------------------');
            return 0;
        }

        $jobsArguments = [];

        $i = 0;
        foreach ($storesList as $storeId) {
            $jobsArguments[$i++ % $jobs][] = $storeId;
        }

        foreach ($jobsArguments as $job => $argument) {
            $this->executorGroup->addPhpCommand(
                sprintf('%s %s %s %s', 'bin/magento', $this->getName(), '-s', implode(',', $argument))
            );
        }

        $this->executorGroup->executeAndWaitAll();
        $this->logger->debug('-------------------------END COMMAND--------------------------');
    }

    /**
     * @param string $storesId comma separate store ids
     * @return array
     */
    protected function getStoresToProcess($storesId)
    {
        $allStores = $this->getAllStoreIds();

        if ($storesId === null) {
            return $allStores;
        }

        $stores = explode(',', $storesId);

        $output = [];
        foreach ($stores as $store) {
            $storeId = (int)$store;
            $output[$storeId] = $allStores[$storeId];
        }
        return $output;
    }

    /**
     * @return array
     */
    public function getAllStoreIds()
    {
        $result = [];

        $sql = $this->resource->getConnection()->select()
            ->from($this->resource->getTableName('store'), ['store_id', 'code'])
            ->order('store_id', 'ASC');

        foreach ($this->resource->getConnection()->fetchAll($sql) as $row) {
            if (isset($row['store_id']) && (int)$row['store_id'] > 0) {
                $result[(int)$row['store_id']] = $row['store_id'];
            }
        }

        return $result;
    }

    /**
     * @return array
     */
    public function getAttributesName()
    {
        $result = [];

        $sql = $this->resource->getConnection()->select()
            ->from('eav_attribute')
            ->where('backend_type IN ("text", "varchar")');

        foreach ($this->resource->getConnection()->fetchAll($sql) as $row) {
            $result[(int)$row['attribute_id']] = $row['attribute_code'];
        }

        $result[] = 'entity_id';

        return $result;
    }

    /**
     * @param $storeId
     * @param $allVisible
     * @param $productIds
     * @param $from
     * @param OutputInterface|null $output
     * @param $useDefault
     * @param $attributesNames
     * @return void
     */
    public function regenerateUrlKey(
        $storeId,
        $allVisible,
        $productIds,
        $from,
        OutputInterface $output = null,
        $useDefault = true,
        $attributesNames = []
    ) {
        $collection = $this->collectionFactory->create();

        $collection->setStore($storeId);

        if (!empty($productIds)) {
            $collection->addIdFilter($productIds);
        }

        $collection->addAttributeToSelect(array_merge(['name', 'url_key'], $attributesNames));
        $collection->addStoreFilter($storeId)->setStoreId($storeId);

        if ($allVisible) {
            $collection->addAttributeToFilter('status', ['in' => $this->productStatus->getVisibleStatusIds()]);
            $collection->setVisibility($this->productVisibility->getVisibleInSiteIds());
        }
        if ($from) {
            $collection->addAttributeToFilter('entity_id', ['gteq' => $from]);
        }

        $generated = false;

        $productCount = 0;
        foreach ($collection->getItems() as $product) {
            $errorFlag = false;

            if (!$product->getName()) {
                $this->logger->error('Name is empty for product ' . $product->getId() . ' in store view ' . $storeId);
                continue;
            }
            $name = $product->getName();

            if ($useDefault) {
                $urlKey = $product->formatUrlKey($name . '-' . $storeId . '-' . $product->getId());
            } else {
                $attributes = $product->getData();

                $filteredAttributes = [];

                foreach ($attributesNames as $attributeName) {
                    if (isset($attributes[$attributeName]) && $attributes[$attributeName]) {
                        $filteredAttributes[$attributeName] = $attributes[$attributeName];
                    }
                    if ($attributeName === 'store_code') {
                        $filteredAttributes['store_code'] = $this->helperData->getStoreCode($storeId);
                    }
                }

                foreach ($filteredAttributes as $key => &$attributeName) {
                    if ($attributeName === 'entity_id') {
                        $attributeName = $product->getId();
                    }
                }

                if (in_array($name,$filteredAttributes)) {
                    unset($filteredAttributes['name']);
                }

                $attribute = str_replace(' ', '-', strtolower(implode('-', $filteredAttributes)));
                $urlKey = $product->formatUrlKey($name . '-' . $attribute);
            }

            $oldUrl = $product->getData('url_key');

            $this->oldUrls[$storeId][$product->getId()] = $oldUrl;
            $this->urlKeys[$storeId][$product->getId()] = $urlKey;

            $attributesData = ['url_key' => $urlKey];

            try {
                $this->action->updateAttributes([$product->getId()], $attributesData, $storeId);
                $this->logger->info('Processed urlkey for product ' . $product->getId() . ' storeID ' . $storeId);
                $productCount++;
            } catch (Exception $e) {
                if ($output) {
                    $this->logger->error('Duplicated url for product ' . $product->getId(), $e->getMessage());
                }
            }
        }

        $this->logger->info('Processed ' . $productCount . ' of ' .  count($collection->getItems()) . ' products');
        unset($productCount);
    }

    /**
     * @param $storeId
     * @param $allVisible
     * @param $productIds
     * @param $from
     * @param \Symfony\Component\Console\Output\OutputInterface|null $output
     * @param $useDefault
     * @param $attributeName
     * @return void
     */
    protected function regenerateUrls(
        $storeId,
        $allVisible,
        $productIds,
        $from,
        OutputInterface $output = null,
        $useDefault = true,
        $attributeName = ''

    ) {
        $collection = $this->collectionFactory->create();

        $collection->setStore($storeId);

        if (!empty($productIds)) {
            unset($productIds['pids']);
            $collection->addIdFilter($productIds);
        }

        $collection->addAttributeToSelect(['name', 'url_key', $attributeName]);
        $collection->addStoreFilter($storeId)->setStoreId($storeId);

        if ($allVisible) {
            $collection->addAttributeToFilter('status', ['in' => $this->productStatus->getVisibleStatusIds()]);
            $collection->setVisibility($this->productVisibility->getVisibleInSiteIds());
        }
        if ($from) {
            $collection->addAttributeToFilter('entity_id', ['gteq' => $from]);
        }

        $generated = false;

        $productCount = 0;
        foreach ($collection->getItems() as $product) {
            $errorFlag = false;

            //Remove old redirects
            $this->deleteRedirect($storeId, $product->getId());

            //Generate new base urls
            $product->setStoreId($storeId);

            $this->urlPersist->deleteByData([
                UrlRewrite::ENTITY_ID     => $product->getId(),
                UrlRewrite::ENTITY_TYPE   => ProductUrlRewriteGenerator::ENTITY_TYPE,
                UrlRewrite::REDIRECT_TYPE => 0,
                UrlRewrite::STORE_ID      => $storeId
            ]);

            try {
                $urls = $this->productUrlRewriteGenerator->generate($product);
                $this->urlPersist->replace($urls);
                $this->logger->info('Processed url for product ' . $product->getId() . ' storeID ' . $storeId);
                $productCount++;
            } catch (Exception $e) {
                $this->logger->error('Duplicated url for product ' . $product->getId(), $e->getMessage());
                $errorFlag = true;
            }

            //Generate 301 Redirects
            if (!$this->generate301RedirectConfig) {
                continue;
            }

            $oldUrl = $this->oldUrls[$storeId][$product->getId()];
            $urlKey = $this->urlKeys[$storeId][$product->getId()];

            if ($oldUrl != $urlKey && !$errorFlag) {
                $generated = true;

                $this->generate301Redirect($oldUrl, $urlKey, $storeId, $product->getId());
            }

        }

        $this->logger->info('Processed ' . $productCount . ' of ' .  count($collection->getItems()) . ' products');
        unset($productCount);
    }

    /**
     * @param $storeId
     * @param $productId
     * @return array
     */
    public function deleteRedirect($storeId, $productId)
    {
        $result = [];

        $sql = $this->resource->getConnection()->select()
            ->from('url_rewrite')
            ->where('store_id =' . $storeId)
            ->where('entity_id = ' . $productId)
            ->where('entity_type in ("tools_custom","product")')
            ->where('redirect_type = 301');

     ######## ########    ###    ########   ######    ##          ######## #### ##     ##
        ##    ##         ## ##   ##     ## ##    ##   ##    ##    ##        ##   ##   ##
        ##    ##        ##   ##  ##     ## ##         ##    ##    ##        ##    ## ##
        ##    ######   ##     ## ########   ######    ##    ##    ######    ##     ###
        ##    ##       ######### ##   ##         ##   #########   ##        ##    ## ##
        ##    ##       ##     ## ##    ##  ##    ##         ##    ##        ##   ##   ##
        ##    ######## ##     ## ##     ##  ######          ##    ##       #### ##     ##

        $sqlVarchar = $this->resource->getConnection()->select()
            ->from('catalog_product_entity_varchar','row_id')
            ->join('eav_attribute', 'catalog_product_entity_varchar.attribute_id = eav_attribute.attribute_id', 'eav_attribute.attribute_id'  )
            ->where('eav_attribute.attribute_code = "url_path" ')
            ->where('eav_attribute.entity_type_id = 4')
            ->where('catalog_product_entity_varchar.row_id = ' . $productId);

        foreach ($this->resource->getConnection()->fetchAll($sql) as $row) {
            $this->resource->getConnection()->delete(
                'url_rewrite',
                'entity_id = ' . $row['entity_id'] . " AND redirect_type = 301 AND store_id = {$storeId}"
            );
        }
        foreach ($this->resource->getConnection()->fetchAll($sqlVarchar) as $rowVarchar) {
            $this->resource->getConnection()->delete(
                'catalog_product_entity_varchar',
                'attribute_id = ' . $rowVarchar['attribute_id'] ." AND row_id = {$productId}"
            );
        }

        return $result;
    }

    /**
     *
     */
    protected function configure()
    {
        $this->setName('tools:regenerateurlkey')
            ->setDescription('Regenerate urlkey for given products')
            ->addArgument(
                'pids',
                InputArgument::IS_ARRAY,
                'Products to regenerate'
            )
            ->addOption(
                'store',
                's',
                InputOption::VALUE_REQUIRED,
                'Use the specific Store View'
            )
            ->addOption(
                'all_visible',
                'a',
                InputOption::VALUE_OPTIONAL,
                'Use for all product visible'
            )->addOption(
                'jobs',
                'j',
                InputOption::VALUE_OPTIONAL,
                'Jobs',
                1
            )->addOption(
                'from',
                'from',
                InputOption::VALUE_OPTIONAL,
                'Use for all product from'
            );
    }

    /**
     * @param $oldUrl
     * @param $newUrl
     * @param $storeId
     * @param $productId
     * @return void
     */
    private function generate301Redirect($oldUrl, $newUrl, $storeId, $productId)
    {
        $urlRewriteModel = $this->urlRewriteFactory->create();
        $urlRewriteModel->setStoreId($storeId);
        $urlRewriteModel->setIsSystem(0);
        $urlRewriteModel->setRedirectType('301');
        $urlRewriteModel->setEntityType('product');
        $urlRewriteModel->setEntityId($productId);
        $urlRewriteModel->setTargetPath($newUrl . '.html');
        $urlRewriteModel->setRequestPath($oldUrl . '.html');

        try {
            $urlRewriteModel->save();
            $this->logger->info('Created redirect for product ' . $productId);
        } catch (Exception $e) {
            $this->logger->error('Error with redirect from: ' . $newUrl . ' '.  $e->getMessage());
        }
    }
}

