<?php

namespace FiloBlu\ProductUrlTools\Console\Command;

use Exception;
use Magento\Catalog\Model\Product\Attribute\Source\Status;
use Magento\Catalog\Model\Product\Visibility;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory;
use Magento\CatalogUrlRewrite\Model\ProductUrlRewriteGenerator;
use Magento\Framework\App\ResourceConnection;
use Magento\UrlRewrite\Model\UrlPersistInterface;
use Magento\UrlRewrite\Service\V1\Data\UrlRewrite;
use Psr\Log\LoggerInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Magento\Eav\Model\Config;

/**
 * Class RegenerateProductUrlCommand
 * @package FiloBlu\ProductUrlTools\Console\Command
 */
class RegenerateProductUrlCommand extends Command
{

    /**
     * @var UrlPersistInterface
     */
    protected $urlPersist;

    /**
     * @var CollectionFactory
     */
    protected $collectionFactory;

    /**
     * @var LoggerInterface
     */
    protected $logger;

    /**
     * @var ResourceConnection
     */
    protected $resource;

    /**
     * @var Visibility
     */
    protected $productVisibility;

    /**
     * @var Status
     */
    protected $productStatus;

    /**
     * @var ProductUrlRewriteGenerator
     */
    protected $productUrlRewriteGenerator;

    /**
     * @var Config
     */
    protected $eavConfig;


    /**
     * RegenerateProductUrlCommand constructor.
     * @param CollectionFactory $collectionFactory
     * @param UrlPersistInterface $urlPersist
     * @param LoggerInterface $logger
     * @param Status $productStatus
     * @param Visibility $productVisibility
     * @param ResourceConnection $resource
     * @param ProductUrlRewriteGenerator $productUrlRewriteGenerator
     * @param Config $eavConfig
     * @param null $name
     */
    public function __construct(
        CollectionFactory $collectionFactory,
        UrlPersistInterface $urlPersist,
        LoggerInterface $logger,
        Status $productStatus,
        Visibility $productVisibility,
        ResourceConnection $resource,
        ProductUrlRewriteGenerator $productUrlRewriteGenerator,
        Config $eavConfig,
        $name = null
    )
    {
        $this->collectionFactory = $collectionFactory;
        $this->urlPersist = $urlPersist;
        $this->logger = $logger;
        $this->productVisibility = $productVisibility;
        $this->productStatus = $productStatus;
        $this->resource = $resource;
        $this->productUrlRewriteGenerator = $productUrlRewriteGenerator;
        $this->eavConfig = $eavConfig;
        parent::__construct($name);
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     * @return int|void|null
     */
    public function execute(InputInterface $input, OutputInterface $output)
    {
        $this->logger->debug('----------------------START URL COMMAND-----------------------');

        $errorFlag = false;
        try {
            if ($this->resetProductUrlPathIfExists() >= 0) {
                $this->logger->info('Reset Product Attribute UrlPath');
            }
        } catch (Exception $e) {
            $this->logger->error($e->getMessage());
            $errorFlag = true;
        }

        $storesList = $this->getStoresToProcess($input->getOption('store'));
        $from = $input->getOption('from');

        $productIds = $input->getArgument('pids');

        foreach ($storesList as $storeId) {

            $collection = $this->collectionFactory->create();
            $collection->addStoreFilter($storeId)->setStoreId($storeId);
            $collection->addAttributeToSelect(['url_path', 'url_key']);
            $collection->setStore($storeId);

            $this->logger->info("Processing store $storeId");
            if (!empty($productIds)) {
                $collection->addIdFilter($productIds);
            }
            if ($from){
                $collection->addAttributeToFilter('entity_id',['gteq' => $from]);
            }
            /*
            TODO: Provide appropriate option on command
            else {
                $collection->addAttributeToFilter('status', ['in' => $this->productStatus->getVisibleStatusIds()]);
                $collection->setVisibility($this->productVisibility->getVisibleInSiteIds());
            }*/

            foreach ($collection->load() as $product) {

                $product->setStoreId($storeId);

                $this->urlPersist->deleteByData([
                    UrlRewrite::ENTITY_ID => $product->getId(),
                    UrlRewrite::ENTITY_TYPE => ProductUrlRewriteGenerator::ENTITY_TYPE,
                    UrlRewrite::REDIRECT_TYPE => 0,
                    UrlRewrite::STORE_ID => $storeId
                ]);

                try {
                    $urls = $this->productUrlRewriteGenerator->generate($product);
                    $this->urlPersist->replace($urls);
                    $this->logger->info('Processed url for product ' . $product->getId() . ' storeID ' . $storeId);
                } catch (Exception $e) {
                    $this->logger->error('Duplicated url for ' . $product->getId() . ' ' . $e->getMessage());
                    $errorFlag = true;
                }
            }
        }
        if (!$errorFlag) {
            $this->logger->info('Reset Product Attribute UrlPath');
        }

        $this->logger->debug('-------------------------END COMMAND--------------------------');
        return 0;
    }

    /**
     * @return bool|int
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function resetProductUrlPathIfExists()
    {
        $attributeId = $this->eavConfig->getAttribute('catalog_product','url_path')->getId() ?? -1;
        if($attributeId)
        {
            return $this->resource->getConnection()->delete('catalog_product_entity_varchar',"attribute_id =$attributeId");
        }

        return 0;
    }

    /**
     * @param $storeId
     * @return array
     */
    protected function getStoresToProcess($storeId)
    {
        $allStores = $this->getAllStoreIds();

        if (!empty($storeId) && isset($allStores[$storeId])) {
            return [
                $storeId => $allStores[$storeId]
            ];
        }

        return $allStores;

    }

    /**
     * @return array
     */
    public function getAllStoreIds()
    {
        $result = [];

        $sql = $this->resource->getConnection()->select()
            ->from($this->resource->getTableName('store'), ['store_id', 'code'])
            ->where('`code` <> ?', 'admin')
            ->order('store_id', 'ASC');

        foreach ($this->resource->getConnection()->fetchAll($sql) as $row) {
            if (isset($row['store_id']) && (int)$row['store_id'] > 0) {
                $result[(int)$row['store_id']] = $row['store_id'];
            }
        }

        return $result;
    }

    /**
     *
     */
    protected function configure()
    {
        $this->setName('tools:regenerateurl')
            ->setDescription('Regenerate url for given products, recommended launch regenerate url key first')
            ->addArgument(
                'pids',
                InputArgument::IS_ARRAY,
                'Products to regenerate'
            )
            ->addOption(
                'store', 's',
                InputOption::VALUE_REQUIRED,
                'Use the specific Store View'
            )->addOption(
                'from', 'from',
                InputOption::VALUE_OPTIONAL,
                'Use for all product from'
            );
    }
}
