<?php

namespace FiloBlu\ProductUrlTools\Console\Command;

use Exception;
use Magento\Framework\App\ResourceConnection;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Helper\Table;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Zend_Db;

/**
 * Class FixCategoryUrlCommand
 * @package FiloBlu\ProductUrlTools\Console\Command
 */
class FixCategoryUrlCommand extends Command
{
    /**
     * @var ResourceConnection
     */
    protected $resource;

    /**
     * FixCategoryUrlCommand constructor.
     * @param ResourceConnection $resource
     * @param string $name
     */
    public function __construct(ResourceConnection $resource, $name = null)
    {
        $this->resource = $resource;
        parent::__construct($name);
    }

    /**
     * Configure the command
     */
    protected function configure()
    {
        $this->setName('tools:fix:category-urls');
        $this->setDescription('Fix duplicates Url\'s for categories');

        $this
            ->addOption(
                'show-duplicates-only',
                null,
                InputOption::VALUE_NONE,
                'Shows only duplicates entries without fixing them'
            );
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $duplicates = $this->findDuplicates();

        if (($duplicatesTotals = count($duplicates)) === 0) {
            $output->writeln('<info>No duplicate rows found in category urls</info>');
            return;
        }

        if ($input->getOption('quiet') === false) {
            $table = new Table($output);
            $table
                ->setHeaders(['category_id', 'product_id', 'url_rewrite_id', '# Duplicates'])
                ->setRows($duplicates);
            $table->render();
        }

        if ($input->getOption('show-duplicates-only') === false) {

            $connection = $this->resource->getConnection();
            $table = $connection->getTableName('catalog_url_rewrite_product_category');

            foreach ($duplicates as $duplicate) {

                $output->write("Fixing entry {$duplicate['category_id']} {$duplicate['product_id']} {$duplicate['url_rewrite_id']} ... ");

                try {
                    $connection->beginTransaction();
                    $connection->delete($table, [
                        'category_id = ?' => $duplicate['category_id'],
                        'product_id = ?' => $duplicate['product_id'],
                        'url_rewrite_id = ?' => $duplicate['url_rewrite_id']
                    ]);
                    $connection->insert($table, [
                        'category_id' => $duplicate['category_id'],
                        'product_id' => $duplicate['product_id'],
                        'url_rewrite_id' => $duplicate['url_rewrite_id']
                    ]);
                    $connection->commit();
                    $output->writeln(' <info>OK</info>');
                } catch (Exception  $exception) {
                    $connection->rollBack();
                    $output->writeln(' <error>ERROR</error>');
                    $output->writeln($exception->getMessage());
                }
            }
        }

        return 0;
    }

    /**
     * @return array
     */
    protected function findDuplicates()
    {
        $connection = $this->resource->getConnection();
        $table = $connection->getTableName('catalog_url_rewrite_product_category');
        return $connection->fetchAll("SELECT * FROM (SELECT `category_id`, `product_id`,`url_rewrite_id`, COUNT(*) AS `duplicates` FROM `{$table}` GROUP BY `category_id`, `product_id`,`url_rewrite_id`) AS `q` WHERE `q`.`duplicates`> 1", [], Zend_Db::FETCH_ASSOC);
    }

}
