<?php
/**
 * UnCancelOrder
 *
 * @package Genmato_UnCancelOrder
 * @author  Vladimir Kerkhoff <support@genmato.com>
 * @created 2015-11-25
 * @copyright Copyright (c) 2015 Genmato BV, https://genmato.com.
 */
namespace Genmato\UnCancelOrder\Observer;

use \Magento\Framework\Event\ObserverInterface;
use \Magento\CatalogInventory\Api\StockManagementInterface;
use \Magento\CatalogInventory\Model\Indexer\Stock\Processor as StockProcessor;
use \Magento\Framework\Event\Observer as EventObserver;
use Magento\InventorySalesApi\Api\Data\SalesChannelInterface;
use Magento\InventorySalesApi\Api\Data\SalesEventInterface;
use \Psr\Log\LoggerInterface;
use \Magento\Framework\Module\Manager;
use \Magento\Framework\App\ResourceConnection;
use \Magento\Store\Model\StoreManagerInterface;
use Magento\Framework\ObjectManagerInterface;

class SubtractInventoryObserver implements ObserverInterface
{
    /**
     * @var StockManagementInterface
     */
    protected $stockManagement;

    /**
     * @var StockProcessor
     */
    protected $stockIndexerProcessor;

    /**
     * @var \Magento\CatalogInventory\Observer\ItemsForReindex
     */
    protected $itemsForReindex;

    /**
     * @var LoggerInterface
     */
    protected $logger;

    /**
     * @var Manager
     */
    protected $_moduleManager;

    protected $connection;

    /**
     * @var StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var ObjectManagerInterface
     */
    protected $_objectManager;

    /**
     * SubtractInventoryObserver constructor.
     * @param StockManagementInterface $stockManagement
     * @param StockProcessor $stockIndexerProcessor
     * @param LoggerInterface $logger
     * @param Manager $moduleManager
     * @param ResourceConnection $resource
     * @param StoreManagerInterface $storeManager
     * @param ObjectManagerInterface $objectManager
     */
    public function __construct(
        StockManagementInterface $stockManagement,
        StockProcessor $stockIndexerProcessor,
        LoggerInterface $logger,
        Manager $moduleManager,
        ResourceConnection $resource,
        StoreManagerInterface $storeManager,
        ObjectManagerInterface $objectManager
    ) {
        $this->stockManagement = $stockManagement;
        $this->stockIndexerProcessor = $stockIndexerProcessor;
        $this->logger = $logger;
        $this->_moduleManager = $moduleManager;
        $this->connection = $resource->getConnection();
        $this->storeManager = $storeManager;
        $this->_objectManager = $objectManager;
    }

    /**
     * Subtract items qtys from stock related with uncancel products.
     *
     * @param EventObserver $observer
     * @return $this
     */
    public function execute(EventObserver $observer)
    {
        $order = $observer->getEvent()->getOrder();
        $productQty = $observer->getEvent()->getProductQty();

        if ($order->getInventoryProcessed()) {
            return $this;
        }

        /**
         * Reindex items
         */
        $itemsForReindex = $this->stockManagement->registerProductsSale(
            $productQty,
            $order->getStore()->getWebsiteId()
        );

        $productIds = [];
        foreach ($itemsForReindex as $item) {
            $item->save();
            $productIds[] = $item->getProductId();
        }
        if (!empty($productIds)) {
            $this->stockIndexerProcessor->reindexList($productIds);
        }

        $order->setInventoryProcessed(true);

        if($this->_moduleManager->isEnabled('Magento_Inventory'))
        {
            $stockResolver = $this->_objectManager->create('\Magento\InventorySalesApi\Api\StockResolverInterface');
            $getSkusByProductIds = $this->_objectManager->create('\Magento\InventoryCatalogApi\Model\GetSkusByProductIdsInterface');
            $getProductTypesBySkus = $this->_objectManager->create('Magento\InventoryCatalogApi\Model\GetProductTypesBySkusInterface');
            $isSourceItemManagementAllowedForProductType = $this->_objectManager->create('\Magento\InventoryConfigurationApi\Model\IsSourceItemManagementAllowedForProductTypeInterface');
            $reservationBuilder = $this->_objectManager->create('\Magento\InventoryReservationsApi\Model\ReservationBuilderInterface');
            $appendReservations = $this->_objectManager->create('\Magento\InventoryReservationsApi\Model\AppendReservationsInterface');
            $serializer = $this->_objectManager->create('\Magento\Framework\Serialize\SerializerInterface');

            $orderId = $order->getId();
            $store = $order->getStore();
            $websiteCode = $this->storeManager->getWebsite($store->getWebsiteId())->getCode();
            $stockId = $stockResolver->execute(SalesChannelInterface::TYPE_WEBSITE, $websiteCode)->getStockId();

            $productSkus = $getSkusByProductIds->execute(array_keys($productQty));
            $productTypes = $getProductTypesBySkus->execute($productSkus);
            $itemsBySku = [];
            foreach ($productSkus as $productId => $sku) {
                if (false === $isSourceItemManagementAllowedForProductType->execute($productTypes[$sku])) {
                    continue;
                }
                $itemsBySku[$sku] = $productQty[$productId];
            }

            $orderEvent = [
                'event_type' => SalesEventInterface::EVENT_ORDER_PLACED,
                'object_type' => SalesEventInterface::OBJECT_TYPE_ORDER,
                'object_id' => $orderId
            ];

            $reservations = [];
            foreach ($itemsBySku as $currentSku => $qty)
            {
                $skuNotExistInCatalog = !isset($productTypes[$currentSku]);
                if ($skuNotExistInCatalog ||
                    $isSourceItemManagementAllowedForProductType->execute($productTypes[$currentSku]))
                {
                    $reservations[] = $reservationBuilder
                        ->setSku($currentSku)
                        ->setQuantity((float)($qty * -1))
                        ->setStockId($stockId)
                        ->setMetadata($serializer->serialize($orderEvent))
                        ->build();
                }
            }

            //$this->connection->delete('inventory_reservation', new \Zend_Db_Expr("JSON_EXTRACT(metadata, '$.object_id') = '".$orderId."' and JSON_EXTRACT(metadata, '$.event_type') = 'order_canceled'"));
            $this->connection->delete('inventory_reservation', new \Zend_Db_Expr("JSON_EXTRACT(metadata, '$.object_id') = '".$orderId."'"));
            $appendReservations->execute($reservations);
            /*$now = new \DateTime();
            $this->connection->update(
                'inventory_reservation',
                ['note' => new \Zend_Db_Expr("'Order uncaceled at ".$now->format('Y-m-d H:i:s')."'")],
                new \Zend_Db_Expr("JSON_EXTRACT(metadata, '$.object_id') = '".$orderId."' and JSON_EXTRACT(metadata, '$.event_type') = '".SalesEventInterface::EVENT_ORDER_PLACED."'")
            )*/
        }

        return $this;
    }
}