<?php

namespace FiloBlu\ThresholdTaxes\Model\ResourceModel;

use FiloBlu\ThresholdTaxes\Helper\Data as ThresholdHelper;
use FiloBlu\ThresholdTaxes\Model\QuoteLocator;
use Magento\Framework\Model\ResourceModel\Db\Context;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Tax\Helper\Data;
use Zend_Db_Expr;

use function is_array;
use function sprintf;

/**
 *
 */
class Calculation extends \Magento\Tax\Model\ResourceModel\Calculation
{
    /**
     * @var ThresholdHelper
     */
    protected $thresholdHelper;

    /**
     * @var \FiloBlu\ThresholdTaxes\Model\QuoteLocator
     */
    private $quoteLocator;

    /**
     * @param Context $context
     * @param Data $taxData
     * @param StoreManagerInterface $storeManager
     * @param \FiloBlu\ThresholdTaxes\Helper\Data $thresholdHelper
     * @param \FiloBlu\ThresholdTaxes\Model\QuoteLocator $quoteLocator
     * @param string|null $connectionName
     */
    public function __construct(
        Context $context,
        Data $taxData,
        StoreManagerInterface $storeManager,
        ThresholdHelper $thresholdHelper,
        QuoteLocator $quoteLocator,
        $connectionName = null
    ) {
        parent::__construct($context, $taxData, $storeManager, $connectionName);
        $this->quoteLocator = $quoteLocator;
        $this->thresholdHelper = $thresholdHelper;
    }

    /**
     * Returns tax rates for request - either performs SELECT from DB, or returns already cached result
     * Notice that productClassId due to optimization can be array of ids
     *
     * @param \Magento\Framework\DataObject $request
     * @return array
     * @throws \Magento\Framework\Exception\LocalizedException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     * @throws \Zend_Db_Select_Exception
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     * @SuppressWarnings(PHPMD.NPathComplexity)
     * @SuppressWarnings(PHPMD.ExcessiveMethodLength)
     */
    protected function _getRates($request)
    {
        $rates = parent::_getRates($request);

        if (!$this->thresholdHelper->isEnabled()) {
            return $rates;
        }

        $select = $this->getConnection()
            ->select()
            ->from(
                $this->getTable('tax_calculation_rate'),
                [
                    'id' => 'tax_calculation_rate_id',
                    'from_total' => new Zend_Db_Expr(
                        sprintf('COALESCE (from_total,%s)', ThresholdHelper::DEFAULT_FROM_TOTAL)
                    ),
                    'to_total' => new Zend_Db_Expr(
                        sprintf('COALESCE (to_total,%s)', ThresholdHelper::DEFAULT_TO_TOTAL)
                    )
                ]
            )
            ->where('tax_calculation_rate_id IN (?)', array_column($rates, 'tax_calculation_rate_id'));

        $fromTo = $this->getConnection()->fetchAssoc($select);
        $quote = $this->getQuote();

        if (!$quote) {
            return $rates;
        }

        if(isset($quote->getTotals()['subtotal_with_discount'])) {
            $subtotal = $quote->getTotals()['subtotal_with_discount']->getValue();
        }
        else {
            $subtotal = $quote->getTotals()['subtotal']->getValue();
        }

        $cacheKey = $this->getRequestCacheKey($request);
        $this->_ratesCache[$cacheKey] = [];
        $selected = null;
        foreach ($rates as $rate) {
            $id = $rate['tax_calculation_rate_id'];

            if (isset($fromTo[$id])) {
                $rate['from_total'] = $fromTo[$id]['from_total'];
                $rate['to_total'] = $fromTo[$id]['to_total'];
            }
            $this->_ratesCache[$cacheKey][] = $rate;

            if (!isset($rate['from_total'], $rate['to_total'])) {
                continue;
            }

            if ($rate['to_total'] <= 0) {
                continue;
            }

            if ($subtotal < $rate['from_total'] || $subtotal > $rate['to_total']) {
                continue;
            }

            $selected = $rate;
        }

        if ($selected) {
            if ($this->mustRecollectQuote($quote, $selected)) {
                $quote->setTriggerRecollect(1);
            }
            return [$selected];
        }

        return $this->_ratesCache[$cacheKey];
    }

    /**
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function getQuote()
    {
        return $this->quoteLocator->getQuote();
    }

    /**
     * @param $request
     * @return string
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getRequestCacheKey($request)
    {
        $storeId = $this->_storeManager->getStore($request->getStore())->getId();
        $customerClassId = $request->getCustomerClassId();
        $countryId = $request->getCountryId();
        $regionId = $request->getRegionId();
        $postcode = $request->getPostcode();
        $productClassId = $request->getProductClassId();
        $ids = is_array($productClassId) ? $productClassId : [$productClassId];
        foreach ($ids as $key => $val) {
            $ids[$key] = (int)$val;
        }
        $ids = array_unique($ids);
        sort($ids);
        $productClassKey = implode(',', $ids);

        return implode('|', [$storeId, $customerClassId, $productClassKey, $countryId, $regionId, $postcode]);
    }

    /**
     * @param $quote
     * @param $selected
     * @return bool
     */
    public function mustRecollectQuote($quote, $selected)
    {
        foreach ($quote->getAllVisibleItems() as $item) {
            if ((float)$item->getTaxPercent() !== (float)$selected['value']) {
                return true;
            }
        }

        return false;
    }
}
