<?php

namespace FiloBlu\Storelocator\Model;


use FiloBlu\Core\Framework\Model\AbstractSqlRepository;
use Magento\Framework\Api\ExtensibleDataInterface;
use Magento\Framework\Api\Search\SearchCriteriaInterfaceFactory;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Api\SearchResultsInterfaceFactory;
use Magento\Framework\App\ResourceConnection;

/**
 *
 */
class StoreRepository extends AbstractSqlRepository
{
    const TABLE_NAME = 'filoblu_storelocator_stores';
    /**
     * @var StoresFactory
     */
    private $storesFactory;
    /**
     * @var SearchCriteriaBuilder
     */
    private $searchCriteriaBuilder;

    /**
     * @param ResourceConnection $resourceConnection
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param SearchCriteriaInterfaceFactory $searchCriteriaFactory
     * @param SearchResultsInterfaceFactory $searchResultsFactory
     * @param StoresFactory $storesFactory
     */
    public function __construct(
        ResourceConnection $resourceConnection,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        SearchCriteriaInterfaceFactory $searchCriteriaFactory,
        SearchResultsInterfaceFactory $searchResultsFactory,
        StoresFactory $storesFactory
    ) {
        parent::__construct($resourceConnection, $searchCriteriaFactory, $searchResultsFactory);
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->storesFactory = $storesFactory;
    }

    /**
     * @return string
     */
    public function getTable(): string
    {
        return self::TABLE_NAME;
    }

    /**
     * @return string
     */
    public function getIdFieldName(): string
    {
        return 'store_id';
    }

    /**
     * @param array $row
     * @return mixed
     */
    public function fromRow(array $row)
    {
        return $this->storesFactory->create(['data' => $row]);
    }

    /**
     * @param string $storeCode
     * @return false|ExtensibleDataInterface|null
     */
    public function getByStoreCode(string $storeCode)
    {
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter('store_code', $storeCode)
            ->create();

        return current($this->getList($searchCriteria)->getItems()) ?: null;
    }

    /**
     * @param string $storeCode
     * @return int|null
     */
    public function getIdFromCode(string $storeCode)
    {
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter('store_code', $storeCode)
            ->create();
        $store = current($this->getList($searchCriteria)->getItems());

        return $store->getId() ?? null;
    }

    /**
     * @param Stores $store
     * @return Stores
     */
    public function save(Stores $store): Stores
    {
        $connection = $this->resourceConnection->getConnection();
        $table = $connection->getTableName(self::TABLE_NAME);
        $storeId = $store->getId();
        $bindings = [
            Stores::STORE_ID                          => $store->getStoreId(),
            Stores::ENABLED                           => $store->getEnabled(),
            Stores::TITLE                             => $store->getTitle(),
            Stores::IMAGE                             => $store->getImage(),
            Stores::LATITUDE                          => $store->getLatitude(),
            Stores::LONGITUDE                         => $store->getLongitude(),
            Stores::STORE_IDS                         => $store->getStoreIds(),
            Stores::STREET                            => $store->getStreet(),
            Stores::POSTCODE                          => $store->getPostcode(),
            Stores::CITY                              => $store->getCity(),
            Stores::COUNTRY                           => $store->getCountry(),
            Stores::PHONE_1                           => $store->getPhone1(),
            Stores::PHONE_2                           => $store->getPhone2(),
            Stores::FAX                               => $store->getFax(),
            Stores::EMAIL                             => $store->getEmail(),
            Stores::WEBSITE_URL                       => $store->getWebsiteUrl(),
            Stores::ADDITIONAL_TEXT                   => $store->getAdditionalText(),
            Stores::MARKER_IMAGE                      => $store->getMarkerImage(),
            Stores::GMB_CAT                           => $store->getGmbCat(),
            Stores::GMB_LANGUAGE_CODE                 => $store->getGmbLanguageCode(),
            Stores::GMB_LOCATION_NAME                 => $store->getGmbLocationName(),
            Stores::REGION                            => $store->getRegion(),
            Stores::REGION_ID                         => $store->getRegionId(),
            Stores::IS_ENABLED_FOR_PICKUPINSTORE      => $store->getIsEnabledForPickupInStore(),
            Stores::URL_KEY                           => $store->getUrlKey(),
            Stores::STORE_CODE                        => $store->getStoreCode(),
            Stores::IS_ENABLED_SHOP_DETAILS           => $store->getIsEnabledShopDetails(),
            Stores::STORE_EXTERNAL_IFRAME_APPOINTMENT => $store->getStoreExternalIframeAppointment(),
            Stores::ADDITIONAL_TEXT_2                 => $store->getAdditionalText2(),
            Stores::STORE_FILENAME_PREFIX             => $store->getStoreFilenamePrefix(),
            Stores::IS_ENABLED_FOR_RETURN_IN_STORE    => $store->getIsEnabledForReturnInStore(),
            Stores::LAST_UPDATE                       => $store->getLastUpdate()
        ];

        $whereCondition = sprintf('%s = %s', Stores::STORE_ID, $storeId);

        if ($storeId && $connection->fetchOne(
                $connection
                    ->select()
                    ->from($table)
                    ->where($whereCondition)
            )) {
            $connection->update($table, $bindings, $whereCondition);

            return $store;
        }

        $connection->insert($table, $bindings);
        $store->setId($connection->lastInsertId($table));

        return $store;
    }
}
