<?php

namespace FiloBlu\Storelocator\Model;

use FiloBlu\Storelocator\Model\Google\Api;
use FiloBlu\Storelocator\Model\ResourceModel\Stores\CollectionFactory;
use Magento\Directory\Model\CountryFactory;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\Request\Http;
use Magento\Framework\Data\Collection\AbstractDb;
use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Model\Context;
use Magento\Framework\Model\ResourceModel\AbstractResource;
use Magento\Framework\Registry;
use Magento\Framework\UrlInterface;
use Magento\Store\Model\StoreManagerInterface;

use function count;

/**
 *
 */
class Stores extends AbstractModel
{
    /** @var string */
    const STORE_ID = 'store_id';
    /** @var string */
    const ENABLED = 'enabled';
    /** @var string */
    const TITLE = 'title';
    /** @var string */
    const IMAGE = 'image';
    /** @var string */
    const LATITUDE = 'latitude';
    /** @var string */
    const LONGITUDE = 'longitude';
    /** @var string */
    const STORE_IDS = 'store_ids';
    /** @var string */
    const STREET = 'street';
    /** @var string */
    const POSTCODE = 'postcode';
    /** @var string */
    const CITY = 'city';
    /** @var string */
    const COUNTRY = 'country';
    /** @var string */
    const PHONE_1 = 'phone_1';
    /** @var string */
    const PHONE_2 = 'phone_2';
    /** @var string */
    const FAX = 'fax';
    /** @var string */
    const EMAIL = 'email';
    /** @var string */
    const WEBSITE_URL = 'website_url';
    /** @var string */
    const ADDITIONAL_TEXT = 'additional_text';
    /** @var string */
    const MARKER_IMAGE = 'marker_image';
    /** TODO: not used
     * @var string
     */
    const gmb_cat = 'gmb_cat';
    /** @var string */
    const GMB_CAT = 'gmb_cat';
    /** @var string */
    const GMB_LANGUAGE_CODE = 'gmb_language_code';
    /** @var string */
    const GMB_LOCATION_NAME = 'gmb_location_name';
    /** @var string */
    const REGION = 'region';
    /** @var string */
    const REGION_ID = 'region_id';
    /** @var string */
    const IS_ENABLED_FOR_PICKUPINSTORE = 'is_enabled_for_pickup_in_store';
    /** @var string */
    const URL_KEY = 'url_key';
    /** @var string */
    const STORE_CODE = 'store_code';
    /** @var string */
    const IS_ENABLED_SHOP_DETAILS = 'is_enabled_shop_details';
    /** @var string */
    const STORE_EXTERNAL_IFRAME_APPOINTMENT = 'store_external_iframe_appointment';
    /** @var string */
    const ADDITIONAL_TEXT_2 = 'additional_text_2';
    /** @var string */
    const STORE_FILENAME_PREFIX = 'store_filename_prefix';
    /** @var string */
    const IS_ENABLED_FOR_RETURN_IN_STORE = 'is_enabled_for_return_in_store';
    /** @var string */
    const LAST_UPDATE = 'last_update';
    /**
     * @var \FiloBlu\Storelocator\Model\ResourceModel\Stores\CollectionFactory
     */
    protected $storesCollectionFactory;

    /**
     * Store manager
     *
     * @var StoreManagerInterface
     */
    protected $storeManager;
    /**
     * @var \FiloBlu\Storelocator\Model\Google\Api
     */
    protected $googleApi;
    /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface|null
     */
    protected $config = null;
    /**
     * @var \Magento\Framework\App\Request\Http
     */
    protected $request;
    /**
     * @var \FiloBlu\Storelocator\Model\WorkingTimesFactory
     */
    private $workingTimesFactory;
    /**
     * @var \Magento\Directory\Model\CountryFactory
     */
    private $countryFactory;

    /**
     * @param \FiloBlu\Storelocator\Model\ResourceModel\Stores\CollectionFactory $storesCollectionFactory
     * @param \Magento\Store\Model\StoreManagerInterface $_storeManager
     * @param \FiloBlu\Storelocator\Model\Google\Api $googleApi
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $config
     * @param \Magento\Framework\Model\Context $context
     * @param \Magento\Framework\Registry $registry
     * @param \Magento\Framework\App\Request\Http $request
     * @param \FiloBlu\Storelocator\Model\WorkingTimesFactory $workingTimesFactory
     * @param \Magento\Directory\Model\CountryFactory $countryFactory
     * @param \Magento\Framework\Model\ResourceModel\AbstractResource|null $resource
     * @param \Magento\Framework\Data\Collection\AbstractDb|null $resourceCollection
     * @param array $data
     */
    public function __construct(
        CollectionFactory $storesCollectionFactory,
        StoreManagerInterface $_storeManager,
        Api $googleApi,
        ScopeConfigInterface $config,
        Context $context,
        Registry $registry,
        Http $request,
        WorkingTimesFactory $workingTimesFactory,
        CountryFactory $countryFactory,
        AbstractResource $resource = null,
        AbstractDb $resourceCollection = null,
        array $data = []
    ) {
        parent::__construct($context, $registry, $resource, $resourceCollection, $data);
        $this->storesCollectionFactory = $storesCollectionFactory;
        $this->storeManager = $_storeManager;
        $this->googleApi = $googleApi;
        $this->config = $config;
        $this->request = $request;
        $this->workingTimesFactory = $workingTimesFactory;
        $this->countryFactory = $countryFactory;
    }

    /**
     * @return string[]
     */
    public function getAvailableStatuses()
    {
        return [
            0 => 'Disabled',
            1 => 'Enabled'
        ];
    }

    /**
     * Processing object before save data
     *
     * @return $this
     */
    public function beforeSave()
    {
        if (is_array($this->getData('store_ids'))) {
            $this->setData('store_ids', implode(',', $this->getData('store_ids')));
        }
        return parent::beforeSave();
    }

    /**
     * Processing object before delete data
     *
     * @return $this
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function beforeDelete()
    {
        if ($this->getGmbLocationName() && $this->config->getValue(
                'filoblu_storelocator_section/google_my_business/gmb_enabled'
            )) {
            $this->googleApi->deleteLocation($this->getGmbLocationName());
        }
        return parent::beforeDelete();
    }

    /**
     * @param $country
     * @return \FiloBlu\Storelocator\Model\ResourceModel\Stores\Collection
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getAllLatLngByCountry($country)
    {
        $collection = $this->getAllStores();
        $collection->addFieldToFilter('country', ['eq' => $country]);
        $collection->addFieldToSelect(['latitude', 'longitude']);

        return $collection;
    }

    /**
     * @return \FiloBlu\Storelocator\Model\ResourceModel\Stores\Collection
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getAllStores()
    {
        $params = $this->request->getParams();

        $collection = $this->storesCollectionFactory->create();
        $collection->addFieldToFilter('enabled', ['eq' => 1]);
        $collection->addStoreFilter($this->storeManager->getStore()->getId());

        if (isset($params['pickup_in_store']) ? (bool)$params['pickup_in_store'] : false) {
            if ($filterStoreCountry = $this->config->getValue('carriers/pickupinstore/filterstorecountry', 'store')) {
                $allowedCountries = explode(',', $filterStoreCountry);

                $collection->addFieldToFilter('country', ['in' => $allowedCountries]);
            }

            $collection->addFieldToFilter('is_enabled_for_pickup_in_store', ['eq' => 1]);
        }

        if (isset($params['return_in_store']) ? (bool)$params['return_in_store'] : false) {
            $collection->addFieldToFilter('is_enabled_for_return_in_store', ['eq' => 1]);
        }

        $collection->setOrder('country', 'ASC');
        $collection->setOrder('city', 'ASC');

        return $collection;
    }

    /**
     * @param $city
     * @param $country
     * @return \FiloBlu\Storelocator\Model\ResourceModel\Stores\Collection
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getAllLatLngByCity($city, $country = null)
    {
        $collection = $this->getAllStores();
        $collection->addFieldToFilter('city', ['eq' => $city]);
        if ($country) {
            $collection->addFieldToFilter('country', ['eq' => $country]);
        }
        $collection->addFieldToSelect(['latitude', 'longitude']);

        return $collection;
    }

    /**
     * @param $country
     * @return string
     */
    public function getCitiesByCountryHTML($country)
    {
        $collection = $this->getCitiesByCountry($country);

        $citiesArray = [];
        $citiesHTML = '';

        foreach ($collection->getData() as $city) {
            if ($city['city'] != '') {
                $citiesArray[] = ['label' => $city['city'], 'value' => $city['city']];
            }
        }

        // Obtain a list of columns
        foreach ($citiesArray as $key => $row) {
            $label[$key] = $row['label'];
        }

        // Sort the data with volume descending, edition ascending
        // Add $data as the last parameter, to sort by the common key
        array_multisort($label, SORT_ASC, $citiesArray);

        array_unshift($citiesArray, ['value' => '', 'label' => __('Select a city...')->__toString()]);

        foreach ($citiesArray as $key => $row) {
            $citiesHTML .= '<option value="' . $row['value'] . '">' . $row['label'] . '</option>';
        }

        return $citiesHTML;
    }

    /**
     * @param $country
     * @return \FiloBlu\Storelocator\Model\ResourceModel\Stores\Collection
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getCitiesByCountry($country)
    {
        $collection = $this->getAllStores();
        $collection->addFieldToSelect('city');
        $collection->addFieldToFilter('country', ['eq' => $country]);
        $collection->getSelect()->group('city');
        return $collection;
    }

    /**
     * @param $region
     * @return string
     */
    public function getCitiesByRegionHTML($region)
    {
        $collection = $this->getCitiesByRegion($region);

        $citiesArray = [];
        $citiesHTML = '';

        foreach ($collection->getData() as $city) {
            if ($city['city'] != '') {
                $citiesArray[] = ['label' => $city['city'], 'value' => $city['city']];
            }
        }

        // Obtain a list of columns
        foreach ($citiesArray as $key => $row) {
            $label[$key] = $row['label'];
        }

        // Sort the data with volume descending, edition ascending
        // Add $data as the last parameter, to sort by the common key
        array_multisort($label, SORT_ASC, $citiesArray);

        array_unshift($citiesArray, ['value' => '', 'label' => __('Select a city...')->__toString()]);

        foreach ($citiesArray as $key => $row) {
            $citiesHTML .= '<option value="' . $row['value'] . '">' . $row['label'] . '</option>';
        }

        return $citiesHTML;
    }

    /**
     * @param $region
     * @return \FiloBlu\Storelocator\Model\ResourceModel\Stores\Collection
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getCitiesByRegion($region)
    {
        $collection = $this->getAllStores();
        $collection->addFieldToSelect('city');
        $collection->addFieldToFilter('region', ['eq' => $region]);

        $collection->getSelect()->group('city');

        return $collection;
    }

    /**
     * @param $country
     * @return false|string
     */
    public function getRegionsByCountryHTML($country)
    {
        $collection = $this->getRegionsByCountry($country);

        $regionsArray = [];
        $regionsHTML = '';


        if ($collection->getSize()) {
            foreach ($collection->getData() as $region) {
                if ($region['region'] != '') {
                    $regionsArray[] = [
                        'label' => $region['region'],
                        'value' => $region['region']
                    ];
                }
            }

            // Obtain a list of columns
            foreach ($regionsArray as $key => $row) {
                $label[$key] = $row['label'];
            }

            // Sort the data with volume descending, edition ascending
            // Add $data as the last parameter, to sort by the common key
            array_multisort($label, SORT_ASC, $regionsArray);

            array_unshift($regionsArray, ['value' => '', 'label' => __('Select a region...')->__toString()]);

            foreach ($regionsArray as $key => $row) {
                $regionsHTML .= '<option value="' . $row['value'] . '">' . $row['label'] . '</option>';
            }
        } else {
            $regionsHTML = false;
        }

        return $regionsHTML;
    }

    /**
     * @param $country
     * @return \FiloBlu\Storelocator\Model\ResourceModel\Stores\Collection
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getRegionsByCountry($country)
    {
        $collection = $this->getAllStores();
        $collection->addFieldToSelect('region');
        $collection->addFieldToFilter('country', ['eq' => $country]);

        $collection->getSelect()->group('region');

        return $collection;
    }

    /**
     * @return string
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getAllAvailableCountriesHTML()
    {
        $collection = $this->getAllAvailableCountries();

        $countriesArray = [];
        $countriesHTML = '';

        foreach ($collection->getData() as $country) {
            if ($country['country'] != '') {
                $countryData = $this->countryFactory->create()->loadByCode($country['country']);
                $countriesArray[] = ['label' => $countryData->getName(), 'value' => $country['country']];
            }
        }

        // Obtain a list of columns
        foreach ($countriesArray as $key => $row) {
            $label[$key] = $row['label'];
        }

        // Sort the data with volume descending, edition ascending
        // Add $data as the last parameter, to sort by the common key
        if (count($countriesArray)) {
            array_multisort($label, SORT_ASC, $countriesArray);
        }

        array_unshift($countriesArray, ['value' => '', 'label' => __('Select a country...')->__toString()]);

        foreach ($countriesArray as $row) {
            $countriesHTML .= '<option value="' . $row['value'] . '">' . $row['label'] . '</option>';
        }

        return $countriesHTML;
    }

    /**
     * @return \FiloBlu\Storelocator\Model\ResourceModel\Stores\Collection
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getAllAvailableCountries()
    {
        $collection = $this->getAllStores();
        $collection->addFieldToSelect('country');

        $collection->getSelect()->group('country');

        return $collection;
    }

    /**
     * @param $lat1
     * @param $lng1
     * @param $lat2
     * @param $lng2
     * @return \FiloBlu\Storelocator\Model\ResourceModel\Stores\Collection
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getStoresByCurrentZoom($lat1, $lng1, $lat2, $lng2)
    {
        $collection = $this->getAllStores();
        $collection->addFieldToFilter('latitude', ['from' => $lat1, 'to' => $lat2]);
        $collection->addFieldToFilter('longitude', ['from' => $lng1, 'to' => $lng2]);

        $modelWT = $this->workingTimesFactory->create();

        foreach ($collection as $store) {
            if ($store->getImage()) {
                $store->setImageUrl(
                    $this->storeManager->getStore()->getBaseUrl(
                        UrlInterface::URL_TYPE_MEDIA
                    ) . 'filoblu/storelocator/stores/image' . $store->getImage()
                );
            }

            //book an appointment
            if ($store->getUrlKey()) {
                $store->setShopViewUrl($this->storeManager->getStore()->getBaseUrl() . $store->getUrlKey());
            } else {
                $store->setShopViewUrl(
                    $this->storeManager->getStore()->getBaseUrl() . 'storelocator/index/shop/id/' . $store->getStoreId()
                );
            }

            //tags
            $tags = $this->getResource()->getTagsIdByStore($store->getStoreId());
            $store->setTagsId($tags);

            $tagsName = $this->getResource()->getTagsNameByStore($store->getStoreId());
            $store->setTagsName($tagsName);

            // image gallery
            $images = $this->getResource()->getImagesStore($store->getStoreId());
            $editImages = [];

            foreach ($images as $key => $img) {
                if ($img['image']) {
                    $editImages[] = [
                        'title'       => $img['title'],
                        'description' => $img['description'],
                        'imageUrl'    => $this->storeManager->getStore()->getBaseUrl(
                                UrlInterface::URL_TYPE_MEDIA
                            ) . 'filoblu/storelocator/stores/image' . $img['image'],
                    ];
                }
            }

            $store->setImageGallery($editImages);

            // working times
            $objWT = $modelWT->loadByStoreId($store->getStoreId());

            if ($objWT->count() == 0) {
                $store->setHasWorkingTimes(false);
            } else {
                $editWorkingTimes = [];
                $checkNothingEdit = false;
                foreach ($objWT as $wt) {
                    $timeFrom = explode(':', explode(' ', $wt['from'])[1]);
                    $fromHH = $timeFrom[0];
                    $fromMM = $timeFrom[1];

                    $wt['from'] = $fromHH . ':' . $fromMM;

                    $timeTo = explode(':', explode(' ', $wt['to'])[1]);
                    $toHH = $timeTo[0];
                    $toMM = $timeTo[1];

                    $wt['to'] = $toHH . ':' . $toMM;

                    $editWorkingTimes[] = [
                        'day'    => $wt['day'],
                        'from'   => $wt['from'],
                        'to'     => $wt['to'],
                        'closed' => (bool)$wt['closed'],
                        'period' => $wt['period']
                    ];

                    if (($fromHH . $fromMM . $toHH . $toMM != '00000000')) {
                        $checkNothingEdit = true;
                    }
                }

                $store->setHasWorkingTimes($checkNothingEdit);
                $store->setWorkingTimes($editWorkingTimes);
            }
        }

        return $collection;
    }

    /**
     * @return array|mixed
     */
    public function getOldTags()
    {
        if (!$this->getStoreId()) {
            return [];
        }
        $array = $this->getData('old_tags');


        if (!$array) {
            $array = $this->getResource()->getOldTags($this);
            $this->setData('old_tags', $array);
        }

        return $array;
    }

    /**
     * @return array|mixed
     */
    public function getImagesPosition()
    {
        if (!$this->getStoreId()) {
            return [];
        }
        $array = $this->getData('images_position');
        if (is_null($array)) {
            $array = $this->getResource()->getImagesPosition($this);
            $this->setData('images_position', $array);
        }

        return $array;
    }

    /**
     * Initialize resource model
     *
     * @return void
     * @noinspection MagicMethodsValidityInspection
     */
    protected function _construct()
    {
        $this->_init(ResourceModel\Stores::class);
    }
}
