<?php

namespace FiloBlu\StockaFix\Cron;

use Exception;
use FiloBlu\StockaFix\Model\Report\ProcessorInterfaceFactory;
use FiloBlu\StockaFix\Model\ReportInterface;
use FiloBlu\StockaFix\Model\ResourceModel\Report\CollectionFactory;
use Psr\Log\LoggerInterface;
use Magento\Framework\Stdlib\DateTime\TimezoneInterfaceFactory;
use Magento\Framework\Stdlib\DateTime\TimezoneInterface;

class Export
{
    protected $logger;
    /**
     * @var CollectionFactory
     */
    private $reportCollectionFactory;
    /**
     * @var ProcessorInterfaceFactory
     */
    private $processorFactory;
    /**
     * @var TimezoneInterfaceFactory
     */
    protected $timezoneFactory;

    /**
     * @var TimezoneInterface
     */
    protected $timezone;

    private   $connection;

    /**
     * Constructor
     *
     * @param LoggerInterface $logger
     * @param CollectionFactory $reportCollectionFactory
     * @param ProcessorInterfaceFactory $processorFactory
     */
    public function __construct(
        LoggerInterface $logger,
        CollectionFactory $reportCollectionFactory,
        TimezoneInterfaceFactory $timezoneFactory,
        ProcessorInterfaceFactory $processorFactory
    ) {
        $this->logger = $logger;
        $this->reportCollectionFactory = $reportCollectionFactory;
        $this->processorFactory = $processorFactory;
        $this->timezoneFactory = $timezoneFactory;
    }

    /**
     * Execute the cron
     *
     * @return void
     */
    public function execute()
    {
        $collection = $this->reportCollectionFactory->create();
        $this->connection = $collection->getResource()->getConnection();

        $reports = $collection
            ->addFieldToFilter('active', ['eq' => 1])
            ->getItems();

        /** @var ProcessorInterface $processor */
        $processor = $this->processorFactory->create();

        /** @var ReportInterface $report */
        foreach ($reports as $report) {
            if (!$this->canExport($report)) {
                continue;
            }

            try {
                $processor->process($report);
                //$date = $this->timezone->date();
                $date = new \DateTime();
                $date = $date->format('Y-m-d H:i:s');
                $report->setData('schedule_last_execute',$date);
                $report->save();
            } catch (Exception $exception) {
                $this->logger->error($exception);
            }
        }
    }

    /**
     * Check conditions for ability to run report export by cron
     *
     * @param ReportInterface $feed
     * @return int
     */
    public function canExport(ReportInterface $report)
    {
        if(!(bool)$report->getData('active_schedule')){
            return true;
        }

        $result = -1;

        $this->timezone = $this->timezoneFactory->create();

        $currentDayOfWeek  = (int)$this->timezone->date()->format('w');
        $currentDayOfMonth = (int)$this->timezone->date()->format('d');
        $currentDayOfYear  = (int)$this->timezone->date()->format('z');
        $currentHour       = (int)$this->timezone->date()->format('G');
        $currentMinutes    = (int)$this->timezone->date()->format('i');
        $currentTime       = (int)$this->timezone->date()->format('G') * 60 + (int)$this->timezone->date()->format('i');
        $currentYear       = (int)$this->timezone->date()->format('Y');
        $currentMonth      = (int)$this->timezone->date()->format('n');
        $currentWeek      = (int)$this->timezone->date()->format('W');

        $scheduleType = $report->getData('schedule_type');

        if($report->getData('schedule_hour') != null && $report->getData('schedule_minute') != null) {
            $hour = $report->getData('schedule_hour');
            $minute = $report->getData('schedule_minute');

            $scheduleTime = (int)$hour * 60 + (int)$minute;

            $lastRun       = strtotime($report->getData('schedule_last_execute') ?? '');
            $lastDayOfYear = (int)$this->timezone->date($lastRun)->format('z');
            $lastTime      = (int)$this->timezone->date($lastRun)->format('G') * 60 + (int)$this->timezone->date($lastRun)->format('i');
            $lastRunYear   = (int)$this->timezone->date($lastRun)->format('Y');
            $lastRunWeek   = (int)$this->timezone->date($lastRun)->format('W');

            if (!$report->getData('schedule_last_execute')) {
                $lastTime = $currentTime - 25;
            }

            // we run generation minimum day ago. Need run generation
            if ($currentDayOfYear > $lastDayOfYear || $currentYear > $lastRunYear) {
                $lastTime = 0;
            }

            if ($scheduleType == 0) {
                if ($currentTime >= $scheduleTime
                    && $scheduleTime > $lastTime
                    && ($currentTime - $scheduleTime) <= 20
                    && $currentTime - $lastTime > 20
                ) {
                    return true;
                }
            } else if ($scheduleType == 1) {
                if ((!$lastRunWeek || $currentWeek > $lastRunWeek || $currentYear > $lastRunYear) && ($currentTime - $scheduleTime) <= 20) {
                    return true;
                }
            } else if ($scheduleType == 2) {
                if ((!$lastRunYear || $currentYear > $lastRunYear || $currentYear > $lastRunYear) && ($currentTime - $scheduleTime) <= 20) {
                    return true;
                }
            }

        }

        return false;
    }

    /**
     * Check conditions for ability to run report export by cron
     *
     * @param ReportInterface $feed
     * @return int
     */
    public function canExportExpression(ReportInterface $report)
    {
        if(trim($report->getData('schedule_frequency')) == ''){
            return true;
        }

        $result = -1;

        $this->timezone = $this->timezoneFactory->create();

        $currentDayOfWeek  = (int)$this->timezone->date()->format('w');
        $currentDayOfMonth = (int)$this->timezone->date()->format('d');
        $currentDayOfYear  = (int)$this->timezone->date()->format('z');
        $currentHour       = (int)$this->timezone->date()->format('G');
        $currentMinutes    = (int)$this->timezone->date()->format('i');
        $currentTime       = (int)$this->timezone->date()->format('G') * 60 + (int)$this->timezone->date()->format('i');
        $currentYear       = (int)$this->timezone->date()->format('Y');
        $currentMonth      = (int)$this->timezone->date()->format('n');

        $scheduleExpression = $report->getData('schedule_frequency');
        $scheduleExpression = trim($scheduleExpression);

        $scheduleExpressionArray = explode(' ', $scheduleExpression);


        if($this->checkValue($currentDayOfMonth, $scheduleExpressionArray[2])
            && $this->checkValue($currentMonth, $scheduleExpressionArray[3])
            && $this->checkValue($currentDayOfWeek, $scheduleExpressionArray[4])){

            if($this->checkValue($currentDayOfMonth, $scheduleExpressionArray[0])
                && $this->checkValue($currentMonth, $scheduleExpressionArray[1])){
                return true;
            }

            $scheduleTime = (int)$scheduleExpressionArray[1] * 60 + (int)$scheduleExpressionArray[0];
            if(($currentTime - $scheduleTime) <= 20){
                return true;
            }

        }

        return false;
    }

    private function checkValue($current, $expression){

        if($expression == '*'){
            return true;
        }
        else if(strpos($expression, ',') !== false){
            $valeus = explode(',', $expression);
            if(in_array($current, $valeus)){
                return true;
            }
        }
        else if(strpos($expression, '-') !== false){
            $valeus = explode('-', $expression);
            if($current >= $valeus[0] && $current <= $valeus[0]){
                return true;
            }
        }
        else if(strpos($expression, '/') !== false){
            $valeus = explode('/', $expression);
            if(($current % $valeus[1]) == 0){
                return true;
            }
        }
        else if($current == $expression){
            return true;
        }

        return false;
    }
}
