<?php

namespace FiloBlu\StockaFix\Model\Report;

use DateTime;
use FiloBlu\StockaFix\Helper\MailHelper;
use FiloBlu\StockaFix\Model\Report;
use FiloBlu\StockaFix\Model\ReportInterface;
use FiloBlu\StockaFix\Model\ResourceModel\Report\CollectionFactory;
use Magento\Framework\App\Filesystem\DirectoryList;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\DB\Adapter\AdapterInterface;
use Magento\Framework\Exception\FileSystemException;
use Magento\Framework\Exception\MailException;
use Magento\Framework\File\Csv;
use Magento\Framework\Filesystem;
use RuntimeException;
use Zend_Db;
use FiloBlu\StockaFix\Model\ExportStatusFactory;
use FiloBlu\StockaFix\Helper\ReportHelper;

class Processor implements ProcessorInterface
{

    const EXPORT_STATUS_TABLE = 'filoblu_stockafix_export_status';

    /**
     * @var AdapterInterface
     */
    protected $connection;
    /**
     * @var CollectionFactory
     */
    private $reportCollectionFactory;
    /**
     * @var MailHelper
     */
    private $mailHelper;
    /**
     * @var Csv
     */
    private $csvProcessor;
    /**
     * @var DirectoryList
     */
    private $directoryList;
    /**
     * @var Filesystem
     */
    private $filesystem;
    /**
     * @var ReportStatusInterfaceFactory
     */
    private $reportStatusFactory;
    /**
     * @var ResourceConnection
     */
    private $resourceConnection;

    /**
     * @var ExportStatusFactory
     */
    protected $exportStatusFactory;

    /**
     * @var ReportHelper
     */
    private $reportHelper;


    /**
     * @param CollectionFactory $reportCollectionFactory
     * @param MailHelper $mailHelper
     * @param Csv $csvProcessor
     * @param DirectoryList $directoryList
     * @param Filesystem $filesystem
     * @param ReportStatusInterfaceFactory $reportStatusFactory
     * @param ResourceConnection $resourceConnection
     * @param ExportStatusFactory $exportStatusFactory
     * @param ReportHelper $reportHelper
     */
    public function __construct(
        CollectionFactory $reportCollectionFactory,
        MailHelper $mailHelper,
        Csv $csvProcessor,
        DirectoryList $directoryList,
        Filesystem $filesystem,
        ReportStatusInterfaceFactory $reportStatusFactory,
        ResourceConnection $resourceConnection,
        ExportStatusFactory $exportStatusFactory,
        ReportHelper $reportHelper
    ) {
        $this->reportCollectionFactory = $reportCollectionFactory;
        $this->mailHelper = $mailHelper;
        $this->csvProcessor = $csvProcessor;
        $this->directoryList = $directoryList;
        $this->filesystem = $filesystem;
        $this->reportStatusFactory = $reportStatusFactory;
        $this->resourceConnection = $resourceConnection;
        $this->exportStatusFactory = $exportStatusFactory;
        $this->reportHelper = $reportHelper;
    }

    public function canProcess(ReportInterface $report): bool
    {
        return true; // TODO: Implement canProcess() method.
    }

    /**
     * @param ReportInterface $report
     * @return ReportStatusInterface
     * @throws MailException
     */
    public function process(ReportInterface $report): ReportStatusInterface
    {
        $status = $this->reportStatusFactory->create();

        $reportOutput = $this->getConnection()->fetchAll($report->getSql(), [], Zend_Db::FETCH_ASSOC);
        $count = count($reportOutput);

        if($count > 0 || !$report->getExcludeSendZeroRow()) {
            // PRENDI IL DRIVER

            $csv = $this->produceReport($report, $reportOutput);
            if($report->getStatusActive())
                $exportStatusRows = $this->setPreExportStatus($report, $reportOutput);

            $reportFilename = null;
            if($report->getData('schedule_filename')) {
                $reportFilename = str_replace('%d', date('Y_m_d'), $report->getData('schedule_filename'));
            }

            $to = explode(',', $report->getReceiverEmails());
            $this->mailHelper->sendReportMail($to, [$csv], ['report_name' => $report->getName(), 'report_count' => $count, 'report_filename' => $reportFilename]);
            if($report->getStatusActive())
                $this->setAfterExportStatus($report, $exportStatusRows);
        }

        return $status;
    }

    /**
     * @param ReportInterface $report
     * @param array $data
     * @return string
     * @throws FileSystemException
     */
    public function produceReport(ReportInterface $report, $data = [])
    {
        $fileDirectoryPath = $this->directoryList->
            getPath(DirectoryList::VAR_DIR) .
            DIRECTORY_SEPARATOR . 'stockafix' . DIRECTORY_SEPARATOR;

        if (!is_dir($fileDirectoryPath)) {
            if (!mkdir($fileDirectoryPath, 0777, true) && !is_dir($fileDirectoryPath)) {
                throw new RuntimeException(sprintf('Directory "%s" was not created', $fileDirectoryPath));
            }
        }

        $now = new DateTime();
        $fileName = $report->getName() . '_' . $now->format("YmdHis") . '.csv';
        $filePath = $fileDirectoryPath . $fileName;

        if($report->getData('insert_csv_header') && !empty($data)) {
            $csvHeader = $this->produceHeader($report, array_keys($data[0]));
            array_unshift($data, $csvHeader);
        }

        $this->csvProcessor
            ->setEnclosure('"')
            ->setDelimiter(';')
            ->appendData($filePath, $data);

        return $filePath;
    }

    /**
     * @param $report
     * @param $dataKeys
     * @return mixed
     */
    public function produceHeader($report, $dataKeys) {
        $fieldMapping = $this->reportHelper->getReportFieldMapping($report);
        if(!empty($fieldMapping)) {
            foreach($fieldMapping as $mapIndex => $map) {
                foreach($dataKeys as $dataIndex => $field) {
                    if(trim(strtolower($field) ?? '') == trim(strtolower($map['field']) ?? '')) {
                        $dataKeys[$dataIndex] = $map['label'];
                    }
                }
            }
        }
        return $dataKeys;
    }

    /**
     * @param ReportInterface $report
     */
    public function sendReport(ReportStatusInterface $reportStatus)
    {
       $receivers = $reportStatus->getReport()->getReceivers();

       foreach ($receivers as $receiver)
       {
           $receiver->send($reportStatus);
       }
    }

    /**
     * @return AdapterInterface
     */
    public function getConnection()
    {
        if ($this->connection === null) {
            $this->connection = $this->resourceConnection->getConnection();
        }
        return $this->connection;
    }

    /**
     * @param $report
     * @param $outputData
     * @return array
     */
    public function setPreExportStatus($report, $outputData) {
        $export_inserted_data = [];
        if($report->getReportId() && is_array($outputData)) {
            if(count($outputData) > 0) {
                $status_value = ($report->getStatusPreExport()) ? $report->getStatusPreExport() : 0;
                foreach ($outputData as $row => $data) {
                    $exportStatus_item = $this->exportStatusFactory->create()->getCollection()
                        ->addFieldToFilter('report_id', ['eq', $report->getReportId()])
                        ->addFieldToFilter('reportoutput_id', ['eq', $data['reportoutput_id']])
                        ->getFirstItem();

                    if(count($exportStatus_item->getData()) == 0) {
                        $exportStatus = $this->exportStatusFactory->create();
                        $exportStatus_data = [];
                        $exportStatus_data['report_id'] = $report->getReportId();
                        $exportStatus_data['reportoutput_id'] = $data['reportoutput_id'];
                        $exportStatus_data['status'] = $status_value;
                        $exportStatus->setData($exportStatus_data);
                        $exportStatus->save();
                        $export_inserted_data[] = $exportStatus_data;
                    }
                }
            }
        }

        return $export_inserted_data;
    }

    /**
     * @param $report
     * @param $exportStatus_data
     * @return void
     */
    public function setAfterExportStatus($report, $exportStatus_data) {
        if(is_array($exportStatus_data) && count($exportStatus_data) > 0) {
            $status_value = ($report->getStatusAfterExport()) ? $report->getStatusAfterExport() : 1;
            $report_ids = null;
            $reportoutput_ids = [];
            $status_preexport = [];
            foreach ($exportStatus_data as $items => $data) {
                $report_ids[] = $data['report_id'];
                $reportoutput_ids[] = $data['reportoutput_id'];
                $status_preexport[] = $data['status'];
            }

            $collection = $this->exportStatusFactory->create()->getCollection();
            $collection->addFieldToFilter('report_id', ['in' => $report_ids]);
            $collection->addFieldToFilter('reportoutput_id', ['in' => $reportoutput_ids]);
            $collection->addFieldToFilter('status', ['in' => $status_preexport]);
            foreach ( $collection->getItems() as $item) {
                $item->setData('exported_at', date('Y-m-d H:i:s'));
                $item->setData('status', $status_value);
                $item->save();
            }
        }
    }
}
