<?php
declare(strict_types=1);

namespace FiloBlu\Refilo\Remote;

use Generator;

/**
 * Interface DeletedItemFinderInterface
 *
 * Provides methods to manage and discover deleted item IDs for indexers.
 * Implementations should handle the logic for tracking, installing triggers,
 * cleaning, and discovering deleted items in the context of remote indexers.
 *
 * Methods:
 * - getDeletedIds($ids): Returns an array of IDs that have been deleted from the provided list.
 * - install(): Installs any necessary triggers or mechanisms to track deletions.
 * - clean($ids): Cleans up or removes tracking for the specified IDs.
 * - discover(): Discovers and returns deleted items based on the implementation logic.
 */
interface DeletedItemFinderInterface {

    /**
     * Returns an array of IDs that have been deleted from the provided list.
     * @return mixed
     */
    public function getDeletedIds($ids) : array;

    /**
     * Installs any necessary triggers or mechanisms to track deletions.
     * @return mixed
     */
    public function install();

    /**
     * Cleans up or removes tracking for the specified IDs.
     * @param $ids
     * @return mixed
     */
    public function clean($ids);

    /**
     * Discovers and returns deleted items based on the implementation logic.
     * @return mixed
     */
    public function discover(): Generator;
}
