<?php
namespace FiloBlu\Refilo\Model;
use Exception;
use FiloBlu\Refilo\Api\CustomerGroupManagmentInterface;
use Magento\Customer\Api\AccountManagementInterface;
use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Customer\Api\Data\CustomerInterface;
use Magento\Framework\Exception\InputException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Exception\State\InputMismatchException;
use Magento\Quote\Api\CartRepositoryInterface;
use Psr\Log\LoggerInterface;
use Throwable;

/**
 * Class CustomerGroupManagment
 * @package FiloBlu\Refilo\Model
 */
class CustomerGroupManagment implements CustomerGroupManagmentInterface
{
    /**
     * @var CustomerRepositoryInterface
     */
    private $customerRepository;
    /**
     * @var LoggerInterface
     */
    private $logger;
    /**
     * @var CartRepositoryInterface
     */
    private $cartRepository;
    /**
     * @var AccountManagementInterface
     */
    private $accountManagement;

    /**
     * CustomerGroupManagment constructor.
     * @param CustomerRepositoryInterface $customerRepository
     * @param LoggerInterface $logger
     * @param CartRepositoryInterface $cartRepository
     * @param AccountManagementInterface $accountManagement
     */
    public function __construct(
        CustomerRepositoryInterface $customerRepository,
        LoggerInterface $logger,
        CartRepositoryInterface  $cartRepository,
        AccountManagementInterface  $accountManagement){
        $this->customerRepository = $customerRepository;
        $this->logger = $logger;
        $this->cartRepository = $cartRepository;
        $this->accountManagement = $accountManagement;
    }

    /**
     * @param int $customerId
     * @param int $customerGroupId
     * @return CustomerInterface
     * @throws InputException
     * @throws LocalizedException
     * @throws NoSuchEntityException
     * @throws InputMismatchException
     */
    public function updateCustomerGroup(int $customerId, int $customerGroupId)
    {
        $customer = $this->customerRepository->getById($customerId);
        $oldCustomerGroupId = (int)$customer->getGroupId();
        $cart = null;

        try{
            $cart = $this->cartRepository->getActiveForCustomer($customerId,[$customer->getStoreId()]);
        }catch (NoSuchEntityException $e){
            $this->logger->info($e->getMessage());
        }

        if($oldCustomerGroupId !== $customerGroupId){
            $customer->setGroupId($customerGroupId);
            $this->customerRepository->save($customer);
        }

        if ($cart && $customerGroupId !== (int)$cart->getCustomerGroupId()){
            $cart->setCustomer($customer);
            $this->cartRepository->save($cart);
        }

        return $customer;

    }


    /**
     * @param CustomerInterface $customer
     * @param null $password
     * @param string $redirectUrl
     * @param null $customerGroupId
     * @return CustomerInterface
     */
    public function createAccount(CustomerInterface $customer, $password = null, $redirectUrl = '', $customerGroupId = null)
    {
        try{
            if($customerGroupId){
                $customer->setGroupId($customerGroupId);
            }
            return $this->accountManagement->createAccount($customer,$password,$redirectUrl);
        }catch (Exception $e){
            $this->logger->critical($e->getMessage());
        }catch (Throwable $t){
            $this->logger->critical($t->getMessage());
        }
    }

}
