<?php

declare(strict_types=1);

namespace FiloBlu\Refilo\Block\Order\Email\Items;

use DateTime;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Locale\Resolver;
use Magento\Framework\Locale\ResolverInterface;
use Magento\Framework\Stdlib\DateTime\DateTimeFormatterInterface;
use Magento\Framework\Stdlib\DateTime\TimezoneInterface;
use Magento\Framework\View\Element\Block\ArgumentInterface;
use Magento\Sales\Api\Data\OrderItemInterface;
use Magento\Store\Model\ScopeInterface;

/**
 *
 */
class PreorderItem implements ArgumentInterface
{
    /** @var \Magento\Framework\Stdlib\DateTime\DateTimeFormatterInterface */
    protected $dateFormatter;

    /** @var Resolver */
    protected $resolver;
    /**
     * @var ScopeConfigInterface
     */
    private $scopeConfig;
    /**
     * @var \Magento\Framework\Stdlib\DateTime\TimezoneInterface
     */
    private $localeDate;

    /**
     * @param \Magento\Framework\Stdlib\DateTime\DateTimeFormatterInterface $dateFormatter
     * @param \Magento\Framework\Stdlib\DateTime\TimezoneInterface $localeDate
     * @param \Magento\Framework\Locale\ResolverInterface $resolver
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     */
    public function __construct(
        DateTimeFormatterInterface $dateFormatter,
        TimezoneInterface $localeDate,
        ResolverInterface $resolver,
        ScopeConfigInterface $scopeConfig
    ) {
        $this->dateFormatter = $dateFormatter;
        $this->resolver = $resolver;
        $this->scopeConfig = $scopeConfig;
        $this->localeDate = $localeDate;
    }

    /**
     * @param OrderItemInterface $orderItem
     * @return string
     * @throws NoSuchEntityException|\Exception
     */
    public function getShippingDate(OrderItemInterface $orderItem)
    {
        $product = $orderItem->getProduct();

        if (!$product) {
            return '';
        }

        $locale = $this->scopeConfig->getValue(
            $this->resolver->getDefaultLocalePath(),
            ScopeInterface::SCOPE_STORE,
            $orderItem->getStoreId()
        ) ?? $this->resolver->getDefaultLocale();

        $staticDate = new DateTime($product->getDateShipping());

        $format = $this->localeDate->getDateFormat();
        return $this->dateFormatter->formatObject($staticDate, $format, $locale);
    }

    /**
     * @param \Magento\Sales\Api\Data\OrderItemInterface $orderItem
     * @return bool
     */
    public function hasPreorderDate(OrderItemInterface $orderItem): bool
    {
        $product = $orderItem->getProduct();

        if (!$product) {
            return false;
        }

        return ((bool)$product->getIsPreorder()) && $product->getDateShipping();
    }

    /**
     * @param \Magento\Sales\Api\Data\OrderItemInterface $orderItem
     * @return bool
     */
    public function hasPreorderWithoutDate(OrderItemInterface $orderItem): bool
    {
        $product = $orderItem->getProduct();

        if (!$product) {
            return false;
        }

        return ((bool)$product->getIsPreorder()) && !$product->getDateShipping();
    }
}


