<?php

namespace FiloBlu\Refilo\Model;

use FiloBlu\Refilo\Api\WishlistManagementInterface;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Framework\DataObjectFactory;
use Magento\Framework\Exception\AlreadyExistsException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Wishlist\Model\ResourceModel\Wishlist;
use Magento\Wishlist\Model\WishlistFactory;

/**
 *
 */
class WishlistManagement implements WishlistManagementInterface
{
    /** @var WishlistFactory */
    protected $wishlistFactory;

    /** @var Wishlist */
    protected $wishlistResource;

    /** @var ProductRepositoryInterface */
    protected $productRepository;
    /**
     * @var DataObjectFactory
     */
    private $objectFactory;


    public function __construct(
        WishlistFactory $wishlistFactory,
        Wishlist $wishlistResource,
        ProductRepositoryInterface $productRepository,
        DataObjectFactory $objectFactory
    ) {
        $this->wishlistFactory = $wishlistFactory;
        $this->wishlistResource = $wishlistResource;
        $this->productRepository = $productRepository;
        $this->objectFactory = $objectFactory;
    }


    /**
     * @param int $productId
     * @param int $customerId
     * @param mixed $attributeId
     * @param mixed $optionValue
     * @return void
     * @throws AlreadyExistsException
     * @throws LocalizedException
     */
    public function saveProductToWishlist(int $productId, int $customerId, $attributeId = null, $optionValue = null)
    {
        $wishlist = $this->wishlistFactory->create()->loadByCustomerId($customerId, true);
        $buyRequest = null;
        if ($attributeId !== null && $optionValue !== null) {
            $buyRequest = $this->objectFactory->create()->setData('super_attribute', [$attributeId => $optionValue]);
        }
        $wishlist->addNewItem($productId, $buyRequest);
        $this->wishlistResource->save($wishlist);
    }


    /**
     * @param int $productId
     * @param int $customerId
     * @param int $qty
     * @return void
     * @throws AlreadyExistsException
     * @throws LocalizedException
     */
    public function updateProductQtyToWishlist(int $productId, int $customerId, int $qty)
    {
        $wishlist = $this->wishlistFactory->create()->loadByCustomerId($customerId);
        if ($wishlist->getId()) {
            $wishItems = $wishlist->getItemCollection();
            foreach ($wishItems as $item) {
                if ((int)$item->getProductId() === $productId) {
                    $buyRequest = $this->objectFactory->create()->setData('qty', $qty);
                    $wishlist->updateItem($item, $buyRequest);
                    $this->wishlistResource->save($wishlist);
                }
            }
        }
    }

    /**
     * @param int $productId
     * @param int $customerId
     * @return void
     * @throws AlreadyExistsException
     * @throws NoSuchEntityException
     */
    public function deleteProductFromWishlist(int $productId, int $customerId)
    {
        $wishlist = $this->wishlistFactory->create()->loadByCustomerId($customerId);
        if ($wishlist->getId()) {
            $wishItems = $wishlist->getItemCollection();
            foreach ($wishItems as $item) {
                if ((int)$item->getProductId() === $productId) {
                    $item->delete();
                    $this->wishlistResource->save($wishlist);
                }
            }
        }
    }

    /**
     * @param int $customerId
     * @return void
     * @throws AlreadyExistsException
     * @throws NoSuchEntityException
     */
    public function deleteAllProductFromWishlist(int $customerId)
    {
        $wishlist = $this->wishlistFactory->create()->loadByCustomerId($customerId);
        if ($wishlist->getId()) {
            $wishItems = $wishlist->getItemCollection();
            foreach ($wishItems as $item) {
                $item->delete();
            }
            $this->wishlistResource->save($wishlist);
        }
    }
}
