<?php

declare(strict_types=1);

namespace FiloBlu\Refilo\Observer;

use FiloBlu\Refilo\Api\Data\OrderMappingInterfaceFactory;
use FiloBlu\Refilo\Api\OrderMappingRepositoryInterface;
use FiloBlu\Refilo\Helper\OrderMappingHelper;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\Module\Manager;
use Magento\Sales\Api\Data\OrderInterface;
use Psr\Log\LoggerInterface;

/**
 * @class OrderSaveAfterMappingObserver
 * @package FiloBlu\Refilo\Observer
 */
class OrderSaveAfterMappingObserver implements ObserverInterface
{
    /**
     * @var OrderMappingHelper
     */
    private $orderMappingHelper;
    /**
     * @var OrderMappingInterfaceFactory
     */
    private $orderMappingFactory;
    /**
     * @var OrderMappingRepositoryInterface
     */
    private $orderMappingRepository;
    /**
     * @var LoggerInterface
     */
    private $logger;
    /**
     * @var Manager
     */
    private $moduleManager;

    /**
     * @param OrderMappingHelper $orderMappingHelper
     * @param OrderMappingInterfaceFactory $orderMappingFactory
     * @param OrderMappingRepositoryInterface $orderMappingRepository
     * @param LoggerInterface $logger
     * @param Manager $moduleManager
     */
    public function __construct(
        OrderMappingHelper $orderMappingHelper,
        OrderMappingInterfaceFactory $orderMappingFactory,
        OrderMappingRepositoryInterface $orderMappingRepository,
        LoggerInterface $logger,
        Manager $moduleManager
    ) {
        $this->orderMappingHelper = $orderMappingHelper;
        $this->orderMappingFactory = $orderMappingFactory;
        $this->orderMappingRepository = $orderMappingRepository;
        $this->logger = $logger;
        $this->moduleManager = $moduleManager;
    }

    /**
     * @param Observer $observer
     * @return void
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function execute(Observer $observer)
    {
        /** @var OrderInterface $order */
        $order = $observer->getEvent()->getOrder();
        if ($order->getCustomerIsGuest()) {
            return;
        }

        $extensionAttributes = $order->getExtensionAttributes();

        $isInStore = false;
        if ($this->moduleManager->isEnabled('Bcame_Viewide')) {
            $isInStore = (bool)$extensionAttributes->getIsViewideOrder();
        }

        $extensionAttributes->setIsInStore($isInStore);
        $order->setExtensionAttributes($extensionAttributes);

        if (!$order->getExtensionAttributes()->getIsInStore()) {
            return;
        }

        $customerIds = $this->orderMappingHelper->getCustomerIdListByEmail($order->getCustomerEmail());

        foreach ($customerIds as $customerId) {
            $row = $this->orderMappingRepository->get($customerId, $order->getEntityId());
            if ($row) {
                continue;
            }

            $orderMapping = $this->orderMappingFactory->create();
            $orderMapping->setCustomerId($customerId);
            $orderMapping->setOrderId($order->getId());
            $orderMapping->setOrderStoreId($order->getStoreId());
            try {
                $this->orderMappingRepository->save($orderMapping);
            } catch (\Exception $e) {
                $this->logger->error($e->getMessage());
            }
        }
    }
}
