<?php

namespace FiloBlu\Refilo\Model\Config;

use FiloBlu\Core\Framework\Model\Mapper;
use Magento\Framework\DataObject;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Store\Api\Data\StoreInterface;

/**
 * Class StoreConfigProvider
 * @package FiloBlu\Refilo\Model\Config
 */
class StoreConfigProvider extends AbstractConfigProvider
{
    /**
     * @param Mapper $mapper
     * @return Mapper
     */
    protected function setMappings(Mapper $mapper)
    {
        return $mapper
            ->add('store/store_id', 'id', self::DEFAULT, Mapper::INT)
            ->add('store/name', 'name', self::DEFAULT)
            ->add('store/code', 'code', self::DEFAULT)
            ->add('store/code', 'isoCode', self::DEFAULT)
            ->add('website/website_id', 'marketplace_id', self::DEFAULT, Mapper::INT)
            ->add('website/code', 'marketplace', self::DEFAULT)
            ->add('website/name', 'marketplace_name', self::DEFAULT)
            ->add('general/locale', 'locale', [])
            ->add('general/locale/code', 'locale/name', '')
            ->add('currencies', 'currencies', [])
            ->add('configs', 'configs', [])
            ->add('base_url', 'base_url','')
            ->add('store/is_active', 'enabled','0', Mapper::STRING)
            ->add('store/is_visible', 'visible_on_storefront','1', Mapper::STRING);

    }

    /**
     * @param DataObject $configurations
     * @param StoreInterface|null $store
     * @return array
     * @throws NoSuchEntityException
     */
    protected function prepareData(DataObject $configurations, StoreInterface $store = null): array
    {
        $data = $configurations->toArray();

        if ($store) {
            $data['store'] = $store->getData();

            $data['store']['is_visible'] = '1';
            if($configurations->getDataByPath('general/restriction/disable_visibility_on_storefront')){
                $data['store']['is_visible'] = '0';
            }

            $data['website'] = $store->getWebsite()->getData();
        }

        $data['currencies'] = [
            ['code' => $configurations->getDataByPath('currency/options/default')]
        ];

        if ($configurations->getDataByPath('web/secure/use_in_frontend')) {
            $data['base_url'] = $configurations->getDataByPath('web/secure/refilo_storefront_secure_base_url');
        } else {
            $data['base_url'] = $configurations->getDataByPath('web/unsecure/refilo_storefront_unsecure_base_url');
        }

        $billingCountries = $configurations->getDataByPath('general/country/allow') ?? '';
        $data['configs'] =
            [
                'default_currency'   => $configurations->getDataByPath('currency/options/default'),
                'store_information'  => $configurations->getDataByPath('general/store_information') ?? '',
                'billing_countries'  => explode(',', $billingCountries),
                'shipping_countries' => explode(',', $configurations->getDataByPath('refilo_geoip/country/allow_shipping') ?? $billingCountries)
            ];

        $affectedCountries = $configurations->getDataByPath('refilo_geoip/country_store/affected_countries');

        if ($affectedCountries !== null) {
            $data['configs']['navigation_countries'] = explode(',', $affectedCountries ?? '');
        }

        $data['configs']['europe_countries'] = explode(',', $configurations->getDataByPath('general/country/eu_countries') ?? '');
        $data['configs']['default_country'] = $configurations->getDataByPath('general/country/default');

        return $data;
    }
}
