<?php

declare(strict_types=1);

namespace FiloBlu\Refilo\Remote\Entity\Provider;

use FiloBlu\Refilo\Helper\Url;
use FiloBlu\Refilo\Remote\Entity\EntityInterface;
use FiloBlu\Refilo\Remote\Entity\EntityProviderInterface;
use FiloBlu\Refilo\Remote\Entity\Provider\Filter\DisabledEntityFilterInterface;
use FiloBlu\Refilo\Remote\Entity\UrlRewrite;
use FiloBlu\Refilo\Remote\Entity\UrlRewriteFactory;
use LogicException;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Store\Api\Data\StoreInterface;
use Magento\Store\Api\Data\WebsiteInterface;
use Magento\Store\Model\Store;
use Magento\Store\Model\Website;

/**
 *
 */
abstract class AbstractUrlProvider extends AbstractFromQueryProvider
{

    /**
     * @var UrlRewriteFactory
     */
    private $urlRewriteFactory;

    /**
     * @var Url
     */
    private $urlHelper;

    /**
     * @var Store|StoreInterface| null
     */
    private $store;

    /**
     * @var \FiloBlu\Refilo\Remote\Entity\Provider\Filter\DisabledEntityFilterInterface
     */
    private $disabledEntityFilter;

    /**
     * @param ResourceConnection $resourceConnection
     * @param ScopeConfigInterface $scopeConfig
     * @param Url $urlHelper
     * @param UrlRewriteFactory $urlRewriteFactory
     * @param \FiloBlu\Refilo\Remote\Entity\Provider\Filter\DisabledEntityFilterInterface $disabledEntityFilter
     */
    public function __construct(
        ResourceConnection $resourceConnection,
        ScopeConfigInterface $scopeConfig,
        Url $urlHelper,
        UrlRewriteFactory $urlRewriteFactory,
        DisabledEntityFilterInterface $disabledEntityFilter
    ) {
        parent::__construct($resourceConnection, $scopeConfig);
        $this->urlRewriteFactory = $urlRewriteFactory;
        $this->urlHelper = $urlHelper;
        $this->disabledEntityFilter = $disabledEntityFilter;
    }

    /**
     * @return Url
     */
    public function getUrlHelper(): Url
    {
        return $this->urlHelper;
    }

    /**
     * @param StoreInterface $store
     * @return EntityProviderInterface
     */
    public function forStore(StoreInterface $store): EntityProviderInterface
    {
        $this->store = $store;
        return $this;
    }

    /**
     * @param WebsiteInterface| Website $website
     * @return EntityProviderInterface
     */
    public function forWebsite(WebsiteInterface $website): EntityProviderInterface
    {
        throw new LogicException('Website filtering is not supported');
    }

    /**
     * @return StoreInterface|Store|null
     */
    public function getStore()
    {
        return $this->store;
    }

    /**
     * @return bool
     */
    public function hasStore(): bool
    {
        return $this->store !== null;
    }

    /**
     * @param array $row
     * @return EntityInterface
     * @throws NoSuchEntityException
     * @throws LocalizedException
     */
    public function adapt(array $row): EntityInterface
    {
        /** @var UrlRewrite $urlRewrite */
        $urlRewrite = $this->getUrlRewriteFactory()->create(['data' => $row]);
        $urlRewrite->setHreflang($this->urlHelper->getHreflang($urlRewrite));
        $urlRewrite->setCanonical($this->urlHelper->getCanonicalUrl($urlRewrite));
        $urlRewrite->setBreadcrumbs($this->urlHelper->getBreadcrumbs($urlRewrite));
        return $urlRewrite;
    }

    /**
     * @return UrlRewriteFactory
     */
    public function getUrlRewriteFactory(): UrlRewriteFactory
    {
        return $this->urlRewriteFactory;
    }

    /**
     * @return \FiloBlu\Refilo\Remote\Entity\Provider\Filter\DisabledEntityFilterInterface
     */
    public function getDisabledEntityFilter(): DisabledEntityFilterInterface
    {
        return $this->disabledEntityFilter;
    }
}
