<?php

declare(strict_types=1);

namespace FiloBlu\Refilo\Console\Command;

use FiloBlu\Refilo\Model\LimitedStoreManagerFactory;
use FiloBlu\Refilo\Remote\AbstractIndexer;
use Magento\Framework\App\ObjectManager;
use Magento\Indexer\Model\IndexerFactory;
use Magento\Store\Model\StoreManagerInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

/**
 * Class RatingIndex
 * @package FiloBlu\Refilo\Console\Command
 */
class RunIndexer extends Command
{
    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    private $storeManager;
    /**
     * @var \Magento\Indexer\Model\IndexerFactory
     */
    private $indexerFactory;
    /**
     * @var \FiloBlu\Refilo\Model\LimitedStoreManagerFactory
     */
    private $limitedStoreManagerFactory;


    /**
     * @param \Magento\Indexer\Model\IndexerFactory $indexerFactory
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     * @param \FiloBlu\Refilo\Model\LimitedStoreManagerFactory $limitedStoreManagerFactory
     * @param string|null $name
     */
    public function __construct(

        IndexerFactory $indexerFactory,
        StoreManagerInterface $storeManager,
        LimitedStoreManagerFactory $limitedStoreManagerFactory,
        string $name = null
    ) {
        parent::__construct($name);
        $this->storeManager = $storeManager;
        $this->indexerFactory = $indexerFactory;
        $this->limitedStoreManagerFactory = $limitedStoreManagerFactory;
    }

    /**
     * @inheritDoc
     */
    protected function configure()
    {
        $this->setName('refilo:indexer:run');
        $this->addArgument('indexer', InputArgument::REQUIRED);
        $this->setDescription('Run Indexer');

        $this->addOption('store', 's', InputOption::VALUE_REQUIRED);
        parent::configure();
    }

    /**
     * Execute the command
     *
     * @param InputInterface $input
     * @param OutputInterface $output
     *
     * @return int
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     * @throws \Exception
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $storeCode = $input->getOption('store');

        $indexer = $this->indexerFactory->create()
                                        ->load($input->getArgument('indexer'));

        $actionClass = $indexer->getActionClass();


        if (!is_a($actionClass, AbstractIndexer::class, true))
        {
            $output->writeln("<error>$actionClass is not a refilo indexer</error>");
            return 1;
        }

        $arguments = [];
        if ($storeCode)
        {
            $store = $this->storeManager->getStore($storeCode);
            $this->storeManager->setCurrentStore($store);
            $limitedStoreManager = $this->limitedStoreManagerFactory->create();
            $limitedStoreManager->setAllowedStores([$store->getCode() => $store]);
            $arguments = [
                'storeManager' => $limitedStoreManager
            ];

        }

        /** @var \FiloBlu\Refilo\Remote\AbstractIndexer $indexerModel */
        $indexerModel = ObjectManager::getInstance()->create($actionClass, $arguments);
        $indexerModel->executeFull();

        return 0;
    }

}
