<?php

namespace FiloBlu\Refilo\Model\Importer;

use FiloBlu\Core\Model\Configuration;
use FiloBlu\Refilo\Model\Adapter\Mongo;
use FiloBlu\Refilo\Model\WishlistManagement;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Model\Product\Type;
use Magento\ConfigurableProduct\Model\Product\Type\Configurable;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\AlreadyExistsException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Symfony\Component\Console\Input\InputInterface;

/**
 *
 */
class Wishlist extends AbstractImporter
{
    /** @var ResourceConnection  */
    private $connection;
    /** @var WishlistManagement  */
    private $wishlistManagement;
    /** @var ProductRepositoryInterface  */
    private $productRepository;
    /** @var Configurable  */
    private $configurable;

    /**
     * @param \FiloBlu\Refilo\Model\Adapter\Mongo $mongo
     * @param \FiloBlu\Core\Model\Configuration $coreConfiguration
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     * @param \Magento\Framework\App\ResourceConnection $connection
     * @param \FiloBlu\Refilo\Model\WishlistManagement $wishlistManagement
     * @param \Magento\Catalog\Api\ProductRepositoryInterface $productRepository
     * @param \Magento\ConfigurableProduct\Model\Product\Type\Configurable $configurable
     */
    public function __construct(
        Mongo                $mongo,
        Configuration        $coreConfiguration,
        ScopeConfigInterface $scopeConfig,
        ResourceConnection   $connection,
        WishlistManagement $wishlistManagement,
        ProductRepositoryInterface $productRepository,
        Configurable $configurable
    )
    {
        parent::__construct(
            $mongo,
            $coreConfiguration,
            $scopeConfig);
        $this->scopeConfig = $scopeConfig;
        $this->connection = $connection;
        $this->wishlistManagement = $wishlistManagement;
        $this->productRepository = $productRepository;
        $this->configurable = $configurable;
    }

    public function preExecute(InputInterface $input = null)
    {
        // Intentionally left blank
    }

    /**
     * @return iterable
     */
    public function getCollections(): iterable
    {
        yield 'wishlist';
    }

    /**
     * @throws NoSuchEntityException
     * @throws AlreadyExistsException
     * @throws LocalizedException
     */
    public function execute(InputInterface $input = null)
    {
        for ($customerId = 1; $customerId <= $this->getMaxCustomerId(); $customerId++) {
            $wishlist = $this->getConnection()->getById('wishlist', (string)$customerId);
            if ($wishlist){
                $this->wishlistManagement->deleteAllProductFromWishlist($customerId);
                $wishlistItems = $wishlist['items'];
                foreach ($wishlistItems as $item){
                    $product = $this->productRepository->get($item['sku']);
                    if ($product->getTypeId() === Type::TYPE_SIMPLE) {
                        $parentIds = array_values($this->configurable->getParentIdsByChild($product->getId()));
                        $parentId = array_shift($parentIds);
                        if ($parentId){
                            $this->wishlistManagement->saveProductToWishlist($parentId, $customerId);
                        }
                        continue;
                    }
                    $this->wishlistManagement->saveProductToWishlist($product->getId(), $customerId);
                }
            }
        }

    }

    /**
     * @param \Symfony\Component\Console\Input\InputInterface|null $input
     * @return void
     */
    public function postExecute(InputInterface $input = null)
    {
        // Intentionally left blank
    }

    /**
     * @return int
     */
    public function getMaxCustomerId()
    {
        $connection = $this->connection->getConnection();
        $maxIdObj = $connection->fetchOne('SELECT entity_id FROM customer_entity ORDER BY entity_id DESC LIMIT 0, 1');
        return (int)$maxIdObj;
    }
}
