<?php
declare(strict_types=1);

namespace FiloBlu\Refilo\Remote\Entity;

use FiloBlu\Refilo\Helper\PageBuilderConverter;
use Magento\Cms\Api\Data\BlockInterface;
use Magento\Cms\Model\Block;
use Magento\Store\Model\StoreManagerInterface;
use Sabberworm\CSS\Parsing\SourceException;

class MagentoToRefiloCmsBlockAdapter
{
    /**
     * @var PageBuilderConverter
     */
    private $pageBuilderConverter;
    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @param PageBuilderConverter $pageBuilderConverter
     * @param StoreManagerInterface $storeManager
     */
    public function __construct(
        PageBuilderConverter  $pageBuilderConverter,
        StoreManagerInterface $storeManager
    )
    {
        $this->pageBuilderConverter = $pageBuilderConverter;
        $this->storeManager = $storeManager;
    }

    /**
     * @param BlockInterface | Block $block
     * @param CmsBlock $cmsBlock
     * @return CmsBlock
     * @throws SourceException
     */
    public function adapt(BlockInterface $block, CmsBlock $cmsBlock) : CmsBlock
    {
        $storeCode = $block->getStoreCode();
        $id = "block_{$block->getId()}_$storeCode";
        return $cmsBlock
            ->setId($id)
            ->addData([
            'metacode'                 => 'block',
            'store'                    => $storeCode,
            'content_id'               => (int)$block->getId(),
            'url_key'                  => $block->getIdentifier(),
            'idx'                      => $block->getIdentifier(),
            'title'                    => $block->getTitle(),
            'type'                     => 'static_block',
            'status'                   => $block->isActive(),
            'content'                  => $this->pageBuilderConverter->convert($block->getContent()),
            'display_entity'           => $block->getDisplayOn(),
            'display_selection_type'   => $block->getSelectionType(),
            'display_selection_target' => $this->parseTargetSelection($block->getSelectionTarget()),
            'display_container'        => $block->getRefiloLayoutContainer(),
            'display_sort_order'       => (int)$block->getRefiloLayoutOrder(),
            'listing_display_page'     => (int)$block->getListingDisplayPage(),
            'listing_display_position' => (int)$block->getListingDisplayPosition(),
            'listing_columns_spacing'  => (int)$block->getListingColumnsSpacing(),
            'listing_row_spacing'      => (int)$block->getListingRowSpacing(),
            'customer_group'           => $block->getCustomerGroup() !== null ? array_map(
                '\intval',
                explode(',', $block->getCustomerGroup())
            ) : null
        ]);
    }

    /**
     * @param $value
     * @return false|string[]
     */
    public function parseTargetSelection($value)
    {
        $strippedValue = preg_replace('/\s/', '', $value ?? '');
        return explode(',', $strippedValue);
    }

}
