<?php

namespace FiloBlu\Referrals\Helper;

use Magento\Checkout\Model\Session;
use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Helper\Context;
use Magento\Framework\DataObjectFactory;
use Magento\Framework\Registry;
use Magento\Newsletter\Model\Subscriber;
use Magento\Newsletter\Model\SubscriberFactory;
use Magento\Quote\Model\QuoteRepository;
use Magento\SalesRule\Model\CouponFactory;
use Magento\SalesRule\Model\ResourceModel\Coupon\UsageFactory;
use Magento\SalesRule\Model\Rule\CustomerFactory;
use Magento\Checkout\Helper\Cart;
use Magento\Store\Model\StoreManagerInterface;

class Rule extends AbstractHelper
{
    /**
     * @var Core
     */
    protected $referralHelper;
    /**
     * @var UsageFactory
     */
    protected $usageFactory;
    /**
     * @var CouponFactory
     */
    protected $couponFactory;
    /**
     * @var CustomerFactory
     */
    protected $customerFactory;
    /**
     * @var DataObjectFactory
     */
    protected $objectFactory;
    /**
     * @var SubscriberFactory
     */
    protected $subscriberFactory;
    /**
     * @var Cart
     */
    protected $cartHelper;
    /**
     * @var Session
     */
    protected $checkoutSession;
    /**
     * @var QuoteRepository
     */
    private $quoteRepository;
    /**
     * @var StoreManagerInterface
     */
    private $storeManager;
    /**
     * @var Registry
     */
    private $registry;

    /**
     * @param Context $context
     * @param Core $referralHelper
     * @param UsageFactory $usageFactory
     * @param CouponFactory $couponFactory
     * @param CustomerFactory $customerFactory
     * @param DataObjectFactory $objectFactory
     * @param SubscriberFactory $subscriberFactory
     * @param Session $checkoutSession
     */
    public function __construct(
        Context $context,
        Core $referralHelper,
        UsageFactory $usageFactory,
        CouponFactory $couponFactory,
        CustomerFactory $customerFactory,
        DataObjectFactory $objectFactory,
        SubscriberFactory $subscriberFactory,
        Session $checkoutSession,
        QuoteRepository $quoteRepository,
        StoreManagerInterface $storeManager,
        Registry $registry
    )
    {
        $this->referralHelper = $referralHelper;
        $this->usageFactory = $usageFactory;
        $this->couponFactory = $couponFactory;
        $this->customerFactory = $customerFactory;
        $this->objectFactory = $objectFactory;
        $this->subscriberFactory = $subscriberFactory;
        $this->checkoutSession = $checkoutSession;
        $this->quoteRepository = $quoteRepository;
        $this->storeManager = $storeManager;
        $this->registry = $registry;
        parent::__construct($context);

    }

    public function getCurrentQuote()
    {
        $quote = null;

        $storeId = $this->storeManager->getStore()->getWebsiteId();

        if($this->checkoutSession->hasQuote()){
            $quote = $this->checkoutSession->getQuote();
        }else{
            $quoteId = $this->checkoutSession->getData('quote_id_' . $storeId);

            $quote = $this->quoteRepository->get($quoteId);
        }
        return $quote;
    }

    public function getCurrentCustomer() {
        $quote = $this->getCurrentQuote();

        $current_customer = $quote->getCustomer()->getId();

        if(!$current_customer) {
            $current_customer = $this->getSubscriber()->getSubscriberId();
        }

        return $current_customer;
    }

    public function getCurrentCartEmailAddress()
    {
        return $this->getCurrentQuote()->getCustomerEmail();
    }

    public function getCouponInvitationType() {
        return $this->referralHelper->getSysConfigValue($this->referralHelper::COUPON_FIELD_INVITATION_TYPE);
    }

    public function getSubscriber()
    {
        $email = $this->checkoutSession->getData(\FiloBlu\Referrals\Plugin\Email::GUEST_EMAIL_KEY);

        if (!$email) {

            if ($this->getCurrentQuote()->hasData('customer_email')) {
                $email = $this->getCurrentCartEmailAddress();
            } else {
                $this->referralHelper->writeLog('COUPON PLUGIN: Current e-mail is empty, cannot apply rule!');
                return false;
            }

        }

        $this->referralHelper->writeLog("COUPON PLUGIN: Current e-mail is: '$email'");


        /** @var Subscriber $subscriber */
        $subscriber = $this->subscriberFactory->create()->loadByEmail($email);

        return $subscriber;
    }

    public function isCurrentUserAReferral()
    {
        /** @var Subscriber $subscriber */
        $subscriber = $this->getSubscriber();

        if (!$subscriber) {
            return false;
        }

        return (bool)$subscriber->getSubscriberId();
    }


    public function canApplyRule($rule) {

        try {

            $couponCode = $rule->getCode();

            if (!$couponCode)  {
                $this->referralHelper->writeLog("COUPON PLUGIN: Coupon code is empty!");
                return false;
            }

            if (!in_array($rule->getId(), [$this->referralHelper->getSenderRuleId(), $this->referralHelper->getReceiverRuleId()])) {
                $this->referralHelper->writeLog("COUPON PLUGIN: " . sprintf("Rule %s is not a referral rule!", $rule->getId()));
                return false;
            }

            $couponCodeArray = explode(',', $couponCode);
            if(count($couponCodeArray) > 1)
            {
                $couponCode = array_pop($couponCodeArray);
            }

            /** @var \FiloBlu\Referrals\Model\Invitation $invitation */
            $invitation = $this->referralHelper->getInvitationByCouponCode($couponCode);

            if (!$invitation || !$invitation->getId()) {
                $this->referralHelper->writeLog("COUPON PLUGIN: " . sprintf("The invitation for coupon code %s does not exists!", $couponCode));
                return false;
            }

            //if customerId doesn't exist
            if(!$this->registry->registry('fb_referral_customer_id')){
                $customerId = $this->getCurrentCustomer();
                $this->registry->register('fb_referral_customer_id', $customerId);
            }else{
                $customerId = $this->registry->registry('fb_referral_customer_id');
            }

            // This customer $customerId, using this coupon $couponCode, is the sender of the invitation $invitation?
            $customerIsSender = false;
            $customerIsReceiver = false;

            if ($customerId && $invitation->getCustomerId() == $customerId){
                $this->referralHelper->writeLog("COUPON PLUGIN: The customer is sender");
                $customerIsSender = true;
            }

            if ($customerId && !$customerIsSender &&  $invitation->getReferralId() == $customerId) {
                $this->referralHelper->writeLog("COUPON PLUGIN: The customer is receiver");
                $customerIsReceiver = true;
            }

            if ($this->getCouponInvitationType() == 'newsletter') {
                $subscriber = $this->getSubscriber();

                if (!$subscriber) {
                    return false;
                }

                if ($this->isCurrentUserAReferral()) {

                    if (!$subscriber->getSubscriberId()) {
                        $this->referralHelper->writeLog("COUPON PLUGIN: The subscriber does not exists!");
                        return false;
                    }
                }

                if (!$customerIsSender && $invitation->getNewsletterReferralId() && $invitation->getNewsletterReferralId() == $subscriber->getSubscriberId()) {
                    $this->referralHelper->writeLog("COUPON PLUGIN: The customer is receiver 2");
                    $customerIsReceiver = true;
                }
            }

            if(!$customerIsReceiver && !$customerIsSender){
                $this->referralHelper->writeLog("COUPON PLUGIN: The customer_id {$customerId} is NOT the sender or the receiver of the invitation_id {$invitation->getInvitationId()}");
                return false;
            }

            //Now we have all we need to validate this coupon with the user
            $db_field = null;
            if($customerIsSender){
                $db_field = $this->referralHelper->getCouponDbFieldNameForWhoSentTheInvite();
                $this->referralHelper->writeLog("COUPON PLUGIN: The customer_id {$customerId} is the sender of the invitation_id {$invitation->getInvitationId()}");
            }
            if($customerIsReceiver){
                $db_field = $this->referralHelper->getCouponDbFieldNameForWhoGotInvited();
                $this->referralHelper->writeLog("COUPON PLUGIN: The customer_id {$customerId} is the receiver of the invitation_id {$invitation->getInvitationId()}");
            }

            $coupon_to_validate = $invitation->getData($db_field);

            if ($couponCode != $coupon_to_validate){
                $this->referralHelper->writeLog("COUPON PLUGIN: The coupon {$couponCode} does not belong to the customer_id {$customerId}");
                return false;
            }

            return true;
        } catch (\Exception $exception) {
            $this->referralHelper->writeLog("COUPON PLUGIN: Error during he coupon {$rule->getCode()} application");
            throw new \Exception($exception->getMessage());
        }
    }
}
