<?php
declare(strict_types=1);

namespace FiloBlu\Referrals\Helper\Refilo;

use FiloBlu\InvitationExtension\Helper\Invitation;
use FiloBlu\Referrals\Helper\Core;
use FiloBlu\Referrals\Helper\Email;
use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Customer\Api\CustomerRepositoryInterfaceFactory;
use Magento\Invitation\Model\ResourceModel\Invitation\CollectionFactory as InvitationCollectionFactory;
use Magento\SalesRule\Model\ResourceModel\Rule\CollectionFactory;

class CouponHelper
{
    /**
     * @var InvitationCollectionFactory
     */
    private $invitationFactory;
    /**
     * @var Core
     */
    private $coreHelper;
    /**
     * @var CustomerRepositoryInterfaceFactory
     */
    private $customerRepositoryInterfaceFactory;
    /**
     * @var CollectionFactory
     */
    private $ruleCollection;
    /**
     * @var Email
     */
    private $emailHelper;

    /**
     * @param InvitationCollectionFactory $invitationFactory
     * @param Invitation $invitationHelper
     * @param Core $coreHelper
     * @param CustomerRepositoryInterfaceFactory $customerRepositoryInterfaceFactory
     * @param CollectionFactory $ruleCollection
     * @param Email $emailHelper
     */
    public function __construct(
        InvitationCollectionFactory        $invitationFactory,
        Core                               $coreHelper,
        CustomerRepositoryInterfaceFactory $customerRepositoryInterfaceFactory,
        CollectionFactory                  $ruleCollection,
        Email                              $emailHelper
    )
    {
        $this->invitationFactory = $invitationFactory;
        $this->coreHelper = $coreHelper;
        $this->customerRepositoryInterfaceFactory = $customerRepositoryInterfaceFactory;
        $this->ruleCollection = $ruleCollection;
        $this->emailHelper = $emailHelper;
    }

    public function generateCoupon($customerId)
    {
        if (!$customerId) {
            return false;
        }

        $customer = $this->retriveCustomerById($customerId);

        if (!$customer) {
            return false;
        }

        $referralCode = $customer->getCustomAttribute('referral_code');
        if (!$referralCode || !$referralCode->getValue()) {
            return false;
        }

        $invitation = $this->getInvitation($customerId);

        if(!$invitation){
            return false;
        }

        $invitationCoupon = $this->getInvitationCoupons($customer->getStoreId());
        if (count($invitationCoupon) > 0) {
            foreach ($invitationCoupon as $couponField => $couponCode) {
                $invitation->setData($couponField, $couponCode);
            }
            $invitation->save();
        }
        return true;
    }

    public function retriveCustomerById($customerId)
    {
        $customerRepository = $this->customerRepositoryInterfaceFactory->create();

        try {
            $customer = $customerRepository->getById($customerId);
        } catch (\Magento\Framework\Exception\NoSuchEntityException $e) {
            return false;
        }

        return $customer;
    }

    public function getInvitation($customerId)
    {
        $invitationCollection = $this->invitationFactory->create();

        return $invitationCollection->getItemByColumnValue('referral_id', $customerId);
    }

    public function getInvitationCoupons($storeId)
    {
        $invitationCoupons = [];
        $sender_db_field = $this->coreHelper->getCouponDbFieldNameForWhoSentTheInvite();
        $receiver_db_field = $this->coreHelper->getCouponDbFieldNameForWhoGotInvited();

        $this->coreHelper->writeLog('Preparing the coupons!');
        $rule_id = $this->coreHelper->getSysConfigValue(
            'filoblu_referrals/general/ruleidsender',
            'store',
            $storeId
        );

        $rule_loaded = $this->loadSalesRuleById($rule_id);

        if ($rule_loaded && $rule_loaded->getRuleId()) {
            $coupon_code_generated = $this->coreHelper->generateCoupon($rule_loaded->getRuleId());
            if ($coupon_code_generated) {
                $invitationCoupons[$sender_db_field] = $coupon_code_generated;
                $this->coreHelper->writeLog(
                    "Success: generated the coupon {$coupon_code_generated} for " . $sender_db_field
                );
            } else {
                $this->coreHelper->writeLog("Error: Coupon for '$sender_db_field' NOT generated!");
            }
        } else {
            $this->coreHelper->writeLog("Error: Salesrule ID '$rule_id' NOT found!");
        }

        // Generating and saving the coupon code (magento_invitation table) for the user that is invited
        $rule_id = $this->coreHelper->getSysConfigValue(
            'filoblu_referrals/general/ruleidreceiver',
            'store',
            $storeId
        );
        $rule_loaded = $this->loadSalesRuleById($rule_id);
        if ($rule_loaded && $rule_loaded->getRuleId()) {
            $coupon_code_generated = $this->coreHelper->generateCoupon($rule_loaded->getRuleId());
            if ($coupon_code_generated) {
                $invitationCoupons[$receiver_db_field] = $coupon_code_generated;
                $this->coreHelper->writeLog(
                    "Success: generated the coupon {$coupon_code_generated} for $receiver_db_field"
                );
            } else {
                $this->coreHelper->writeLog("Error: Coupon for $receiver_db_field NOT generated!");
            }
        } else {
            $this->coreHelper->writeLog("Error: Salesrule ID $rule_id NOT found!");
        }

        return $invitationCoupons;
    }

    public function loadSalesRuleById($id)
    {
        if (!$id || !is_numeric($id)) {
            return false;
        }

        $ruleCollection = $this->ruleCollection->create();

        $ruleCollection->addFieldToSelect('*');
        $ruleCollection->addFieldToFilter('rule_id', $id);

        if ($ruleCollection->getSize()) {
            return $ruleCollection->getFirstItem();
        }

        return false;

    }

    public function sendEmailToReceiver($referralId)
    {
        $this->emailHelper->notify_invitation_receiver_by_referral_id($referralId);
    }

}
