<?php

namespace FiloBlu\Referrals\Helper;

use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Store\Model\ScopeInterface;

class Email extends AbstractHelper
{
    const XML_PATH_EMAIL_TEMPLATE_INVITE_SENDER = 'filoblu_referrals/general/templateidsender';
    const XML_PATH_EMAIL_TEMPLATE_INVITE_RECEIVER = 'filoblu_referrals/general/templateidreceiver';
    const XML_PATH_EMAIL_SENDER = 'trans_email/ident_general/email';
    const XML_PATH_NAME_SENDER = 'trans_email/ident_general/name';

    protected $storeManager;
    protected $inlineTranslation;
    protected $transportBuilder;
    protected $_referralHelper;

    /**
     * @param \Magento\Framework\App\Helper\Context $context
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     * @param \Magento\Framework\Translate\Inline\StateInterface $inlineTranslation
     * @param \Magento\Framework\Mail\Template\TransportBuilder $transportBuilder
     * @param \FiloBlu\Referrals\Helper\Core $referralHelper
     */
    public function __construct(
        \Magento\Framework\App\Helper\Context $context,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Magento\Framework\Translate\Inline\StateInterface $inlineTranslation,
        \Magento\Framework\Mail\Template\TransportBuilder $transportBuilder,
        \FiloBlu\Referrals\Helper\Core $referralHelper
    )
    {
        parent::__construct($context);
        $this->storeManager = $storeManager;
        $this->inlineTranslation = $inlineTranslation;
        $this->transportBuilder = $transportBuilder;
        $this->_referralHelper = $referralHelper;
    }

    /**
     * Return store configuration value of your template field that which id you set for template
     *
     * @param string $path
     * @param int $storeId
     * @return mixed
     */
    private function getConfigValue($path, $storeId)
    {
        return $this->scopeConfig->getValue(
            $path,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * Return store
     * @return \Magento\Store\Api\Data\StoreInterface
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getStore()
    {
        return $this->storeManager->getStore();
    }

    /**
     * @param $variable
     * @param $receiverInfo
     * @param $templateId
     * @return $this
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function generateTemplate($variable, $receiverInfo, $templateId)
    {
        $this->transportBuilder->setTemplateIdentifier($templateId)
            ->setTemplateOptions(
                [
                    'area' => \Magento\Framework\App\Area::AREA_FRONTEND,
                    'store' => $this->storeManager->getStore()->getId(),
                ]
            )
            ->setTemplateVars($variable)
            ->setFrom($this->emailSender())
            ->addTo($receiverInfo['email'], $receiverInfo['name']);

        return $this;
    }

    /**
     * Return email for sender header
     * @return mixed
     */
    public function emailSender()
    {
        $email_sender = $this->scopeConfig->getValue(
            self::XML_PATH_EMAIL_SENDER,
            ScopeInterface::SCOPE_STORE
        );

        $name_sender = $this->scopeConfig->getValue(
            self::XML_PATH_NAME_SENDER,
            ScopeInterface::SCOPE_STORE
        );

        $email_sender_data = [];
        $email_sender_data['email'] = $email_sender;
        $email_sender_data['name'] = $name_sender;

        return $email_sender_data;
    }

    public function notify_invitation_receiver_by_referral_id($referral_id){
        $invitation_found = $this->_referralHelper->getInvitationByDbField('referral_id',$referral_id,true);
        if ($invitation_found){
            return $this->notify_invitation_receiver($invitation_found);
        }

        return false;
    }

    public function notify_invitation_receiver_by_newsletter_referral_id($referral_id){
        $invitation_found = $this->_referralHelper->getInvitationByDbField('newsletter_referral_id',$referral_id,true);
        if ($invitation_found){
            return $this->notify_invitation_receiver($invitation_found);
        }

        return false;
    }

    /**
     * @throws \Exception
     */
    public function notify_invitation_receiver($invitation){

        // Already sent!
        if($invitation->getInvitationId() && $invitation->getWhoGotInvitedCpSent()){
            $this->_referralHelper->writeLog("INVITATION ID {$invitation->getInvitationId()} Error: coupon_for_who_got_invited already sent!" );
            return true;
        }

        // No coupon? Nothing to do!
        if (!$invitation->getCouponForWhoGotInvited()){
            $this->_referralHelper->writeLog("INVITATION ID {$invitation->getInvitationId()} Error: coupon_for_who_got_invited db field is empty!" );
            return true;
        }

        try {
            $this->_referralHelper->writeLog("INVITATION ID {$invitation->getInvitationId()} Sending the email to the user invited with the coupon code!" );

            $storeId = $this->storeManager->getStore($invitation->getStore())->getStoreId();

            // Info about the sender
            $sender_customer_data = $this->_referralHelper->getCustomerById($invitation->getCustomerId());
            // The user the sender invited
            $receiverInfo = ['name' => 'user', 'email' => $invitation->getEmail()];
            // Generating as $email_data the array with the coupon and the sender fullname
            $email_data = [];
            $email_data['coupon'] = $invitation->getCouponForWhoGotInvited();
            $email_data['sender_fullname'] = $sender_customer_data->getFirstname()." ".$sender_customer_data->getLastname();
            $result = $this->notify_send(self::XML_PATH_EMAIL_TEMPLATE_INVITE_RECEIVER,$storeId,$email_data,$receiverInfo);

            if($result === true){
                $this->_referralHelper->writeLog("INVITATION ID {$invitation->getInvitationId()} Success: Sent coupon to the user!" );
                $invitation->setWhoGotInvitedCpSent(1);
                $invitation->save();
            } else {
                $this->_referralHelper->writeLog("INVITATION ID {$invitation->getInvitationId()} Error: Can't send email with the coupon -> {$result}" );
            }


        } catch (\Exception $e) {
            $this->_referralHelper->writeLog("INVITATION ID {$invitation->getInvitationId()} Error: {$e->getMessage()}" );
            throw new \Exception($e->getMessage());
        }

        return $result;
    }

    public function notify_invitation_sender($invitation){

        // Info about the sender (the user who sent the invite)
        $sender_customer_data = $this->_referralHelper->getCustomerById($invitation->getCustomerId());
        // Info about the user the sender invited  (the user who received the invite)
        $receiver_customer_data = $this->_referralHelper->getCustomerById($invitation->getReferralId());

        $storeId =  $this->storeManager->getStore($invitation->getStoreId())->getStoreId();

        if(!$receiver_customer_data){
            return false;
        }
        if(!$receiver_customer_data->getEmail()){
            return false;
        }

        // No coupon? Nothing to do!
        if (!$invitation->getCouponForWhoSentTheInvite()){
            return true;
        }

        // Who will receive this coupon? The sender of the invite
        $receiverInfo = ['name' => $sender_customer_data->getFirstname(). ' ' .$sender_customer_data->getLastname(), 'email' => $sender_customer_data->getEmail()];

        $email_data = [];
        $email_data['coupon'] = $invitation->getCouponForWhoSentTheInvite();
        $email_data['receiver_fullname'] = $receiver_customer_data->getFirstname()." ".$receiver_customer_data->getLastname();

        $couponInvitationType = $this->_referralHelper->getSysConfigValue($this->_referralHelper::COUPON_FIELD_INVITATION_TYPE);

        if ($couponInvitationType === 'newsletter') {
            $email_data['receiver_fullname'] = $invitation->getCustomerFirstname()." ".$invitation->getCustomerLastname();
        }

        $email_data['sender_fullname'] = $sender_customer_data->getFirstname()." ".$sender_customer_data->getLastname();

        $result = $this->notify_send(self::XML_PATH_EMAIL_TEMPLATE_INVITE_SENDER,$storeId, $email_data,$receiverInfo);

        return $result;
    }

    /**
     * @throws \Magento\Framework\Exception\MailException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    private function notify_send($template_path,$storeId, $email_data = [],$receiverInfo = []){
        try {
            $templateId = $this->getConfigValue($template_path,$storeId);
            $this->inlineTranslation->suspend();
            $this->generateTemplate($email_data, $receiverInfo, $templateId);
            $transport = $this->transportBuilder->getTransport();
            $transport->sendMessage();
            $this->inlineTranslation->resume();

        } catch (\Exception $e) {
            $this->_referralHelper->writeLog("REFERRAL ERROR: Unable to notify user: " . $e->getMessage());
            return false;
            //return $e->getMessage();
            //$this->_logger->info($e->getMessage());
        }

        return true;
    }
}
