<?php
namespace FiloBlu\Paymentscapture\Model;


use FiloBlu\Paymentscapture\Helper\CustomLogger;

/**
 * Cancel stuck orders and eventually restore quote.
 * Use case: pending order with PayPal. They can be canceled after short time.
 * We then try to set its quote active.
 *
 * @author Ermanno Baschiera <ermanno@filoblu.com>
 */
class Orderclean extends \Magento\Framework\Model\AbstractModel {

    const ORDER_AGE_CONFIG_PATH = 'filoblu_paymentscapture/order_clean/order_age';
    const ORDER_STATUSES_CONFIG_PATH = 'filoblu_paymentscapture/order_clean/order_status';
    const ORDER_PAYMENT_METHODS_CONFIG_PATH = 'filoblu_paymentscapture/order_clean/payment_method';
    const CLEANING_ACTIVE_CONFIG_PATH = 'filoblu_paymentscapture/order_clean/active';
    const RESET_COUPON_ACTIVE_CONFIG_PATH = 'filoblu_paymentscapture/order_clean/reset_coupon';

    /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * @var \Magento\Quote\Model\QuoteFactory
     */
    protected $quoteQuoteFactory;

    /**
     * @var \Magento\Sales\Model\ResourceModel\Order\CollectionFactory
     */
    protected $salesResourceModelOrderCollectionFactory;

    /**
     * @var \Magento\Framework\DataObjectFactory
     */
    protected $dataObjectFactory;

    /**
     * @var CustomLogger
     */
    protected $customLogger;

    public function __construct(
        \Magento\Framework\Model\Context $context,
        \Magento\Framework\Registry $registry,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Quote\Model\QuoteFactory $quoteQuoteFactory,
        \Magento\Sales\Model\ResourceModel\Order\CollectionFactory $salesResourceModelOrderCollectionFactory,
        \Magento\Store\Model\WebsiteFactory $storeWebsiteFactory,
        CustomLogger $customLogger,
        \Magento\Framework\DataObjectFactory $dataObjectFactory,
        \Magento\Framework\Model\ResourceModel\AbstractResource $resource = null,
        \Magento\Framework\Data\Collection\AbstractDb $resourceCollection = null,
        array $data = []
    ) {
        $this->dataObjectFactory = $dataObjectFactory;
        $this->scopeConfig = $scopeConfig;
        $this->quoteQuoteFactory = $quoteQuoteFactory;
        $this->salesResourceModelOrderCollectionFactory = $salesResourceModelOrderCollectionFactory;
        $this->storeWebsiteFactory = $storeWebsiteFactory;
        $this->customLogger = $customLogger;
        parent::__construct(
            $context,
            $registry,
            $resource,
            $resourceCollection,
            $data
        );
    }


    public function cleanOrders() {
        /** @var \Magento\Framework\App\ObjectManager $om */
        $om = \Magento\Framework\App\ObjectManager::getInstance();
        /** @var \Magento\Store\Model\StoreManagerInterface|\Magento\Store\Model\StoreManager $storeManager */
        $storeManager = $om->get('Magento\Store\Model\StoreManagerInterface');
        $websites = $storeManager->getWebsites();
        foreach ($websites as $website) {
            if (!$this->scopeConfig->isSetFlag(self::CLEANING_ACTIVE_CONFIG_PATH, \Magento\Store\Model\ScopeInterface::SCOPE_WEBSITE, $website)) {
                continue;
            }
            $orders = $this->_getOrdersCollection($website);
            foreach ($orders as $order) {
                $this->customLogger->orderCleanLog('Processing order #' . $order->getIncrementId());
                $this->_cancel_and_restore_quote($order);
            }
        }
    }

    protected function reactivateCoupon($order)
    {
        if ($rules = $order->getAppliedRuleIds()) {
            $objectManager = \Magento\Framework\App\ObjectManager::getInstance();

            foreach(explode(",", $rules) as $ruleId){
                $rule = $objectManager->create('\Magento\SalesRule\Model\Rule')->load($ruleId);
                $rule->setTimesUsed($rule->getTimesUsed()-1);
                $rule->save();
                if ($rule->getCouponType() == 2 && $couponCode = $order->getCouponCode()) {
                    $coupon = $objectManager->create('\Magento\SalesRule\Model\Coupon')->loadByCode($couponCode);
                    $coupon->setTimesUsed($coupon->getTimesUsed()-1);
                    $coupon->save();
                    if($customerId = $order->getCustomerId()) {
                        $couponUsage = $objectManager->create('\FiloBlu\Paymentscapture\Model\Usage');
                        $couponUsage->reduceCouponTimesUsed($customerId, $coupon->getData('coupon_id'));
                    }
                }

                if($customerId = $order->getCustomerId()) {
                    $ruleCustomer = $objectManager->create('\Magento\SalesRule\Model\Rule\Customer')->loadByCustomerRule($customerId, $ruleId);
                    if ($ruleCustomer->getRuleCustomerId()) {
                        $ruleCustomer->setTimesUsed($ruleCustomer->getTimesUsed()-1);
                        $ruleCustomer->save();
                    }
                }
            }
        }
    }


    /**
     *
     * @param \Magento\Sales\Model\Order $order
     * @return boolean
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function _cancel_and_restore_quote($order) {
        try {
            if (!in_array($order->getPayment()->getMethod(), explode(",", $this->scopeConfig->getValue(self::ORDER_PAYMENT_METHODS_CONFIG_PATH, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $order->getStoreId())))) {
                $this->customLogger->orderCleanLog('Cannot cancel orders for payment code ' . $order->getPayment()->getMethod());
                return FALSE;
            }

            $can_cancel_from_observer = $this->_getCanCancelFromObserver($order);

            if ($can_cancel_from_observer == FALSE) {
                $this->customLogger->orderCleanLog('This order has a payment method which prevents canceling');
                return FALSE;
            }

            if (!$order->canCancel()) {
                throw new \Magento\Framework\Exception\LocalizedException(__('Order cannot be canceled'));
            }

            $order->registerCancellation("Canceled by FiloBlu_Paymentscapture_Model_Orderclean")->save();
            if($this->scopeConfig->isSetFlag(self::RESET_COUPON_ACTIVE_CONFIG_PATH, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $order->getData("store_id"))) {
                $this->reactivateCoupon($order);
            }
            $this->customLogger->orderCleanLog('Order canceled');

            if ($order->getCustomerIsGuest() == TRUE) {
                //quote cannot be restored
                $this->customLogger->orderCleanLog('Quote cannot be restored');
                return TRUE;
            }

            //retrieve current active quote for the customer
            //if it's null, restore the quote from the order
            $customer_quote = $this->quoteQuoteFactory->create()->loadByCustomer($order->getCustomerId());
            if ($customer_quote->getId() != NULL) {
                //customer has a new active quote, so we cannot restore the quote from the order
                $this->customLogger->orderCleanLog('Customer has a new active quote');
                return TRUE;
            }

            $order_quote = $this->quoteQuoteFactory->create()->setStoreId($order->getStoreId())->load($order->getQuoteId());
            if ($order_quote->getId()) {
                $order_quote->setIsActive(1)
                    ->setReservedOrderId(NULL)
                    ->save();
                $this->customLogger->orderCleanLog('Quote restored');
            } else {
                $this->customLogger->orderCleanLog('Could not load quote from order');
            }

        } catch (\Exception $e) {
            $this->customLogger->orderCleanLog('Exception with ' . $order->getIncrementId() . ': ' . $e->getMessage());
            return FALSE;
        }

        return TRUE;
    }

    protected function _getCanCancelFromObserver($order) {
        $eventObject = $this->dataObjectFactory->create();
        $eventObject->setCanCancel(TRUE);
        $eventObject->setOrder($order);

        $this->_eventManager->dispatch('paymentscapture_orderclean_can_cancel', array('event_object' => $eventObject));

        return $eventObject->getCanCancel();
    }

    protected function _getOrdersCollection($website) {
        $order_min_age = $this->scopeConfig->getValue(self::ORDER_AGE_CONFIG_PATH, \Magento\Store\Model\ScopeInterface::SCOPE_WEBSITE, $website);
        if (!is_numeric($order_min_age) || trim($order_min_age) == '') {
            $this->customLogger->orderCleanLog('$order_min_age is not correct');
            return array();
        }

        $upper_limit = new \DateTime('@' . mktime(date("H"), date("i") - $order_min_age, 0, date("m"), date("d"), date("Y")));

        $collection = $this->salesResourceModelOrderCollectionFactory->create()->addAttributeToFilter('store_id', array('in' => $website->getStoreIds()))
            ->addAttributeToFilter('created_at', array('lteq' => $upper_limit->format('Y-m-d H:i:s')));

        $statuses = explode(',', $this->scopeConfig->getValue(self::ORDER_STATUSES_CONFIG_PATH, \Magento\Store\Model\ScopeInterface::SCOPE_WEBSITE, $website));
        if (count($statuses) > 0) {
            $collection->addAttributeToFilter('status', array('in' => $statuses));
        } else {
            return array();
        }

        return $collection;
    }

}
