<?php

namespace FiloBlu\Paymentscapture\Model;

use FiloBlu\Paymentscapture\Helper\CustomLogger;

class Capture extends \Magento\Framework\Model\AbstractModel {

    const ORDER_AGE_CONFIG_PATH = 'filoblu_paymentscapture/general/order_age';
    const ORDER_STATUSES_CONFIG_PATH = 'filoblu_paymentscapture/general/order_status';
    const ORDER_PAYMENT_METHODS_CONFIG_PATH = 'filoblu_paymentscapture/general/payment_method';
    const VOIDED_ORDER_CHECK_CONFIG_PATH = 'filoblu_paymentscapture/general/check_void';
    const CAPTURE_ACTIVE_CONFIG_PATH = 'filoblu_paymentscapture/general/active';
    const ADYEN_HPP_PAYMENT_CODE = 'adyen_hpp';
    const ORDER_ADYENHPP_PAYMENT_METHODS_CONFIG_PATH = 'filoblu_paymentscapture/general/adyenhpp_payment_method';

    protected $_helper;

    /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * @var \Magento\Sales\Model\ResourceModel\Order\CollectionFactory
     */
    protected $salesResourceModelOrderCollectionFactory;

    /**
     * @var \Magento\Framework\DB\TransactionFactory
     */
    protected $transactionFactory;

    /**
     * @var \FiloBlu\Paymentscapture\Helper\Data
     */
    protected $paymentscaptureHelper;

    /**
     * @var \Magento\Sales\Model\Service\InvoiceService
     */
    protected $invoiceService;

    /**
     * @var CustomLogger
     */
    protected $customLogger;

    public function __construct(
    \Magento\Framework\Model\Context $context, \Magento\Framework\Registry $registry, \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig, \Magento\Sales\Model\ResourceModel\Order\CollectionFactory $salesResourceModelOrderCollectionFactory, \Magento\Framework\DB\TransactionFactory $transactionFactory, \FiloBlu\Paymentscapture\Helper\Data $paymentscaptureHelper, \Magento\Sales\Model\Service\InvoiceService $invoiceService, CustomLogger $customLogger, \Magento\Framework\Model\ResourceModel\AbstractResource $resource = null, \Magento\Framework\Data\Collection\AbstractDb $resourceCollection = null, array $data = array()) {
        $this->scopeConfig = $scopeConfig;
        $this->salesResourceModelOrderCollectionFactory = $salesResourceModelOrderCollectionFactory;
        $this->transactionFactory = $transactionFactory;
        $this->paymentscaptureHelper = $paymentscaptureHelper;
        $this->invoiceService = $invoiceService;
        $this->customLogger = $customLogger;
        parent::__construct(
                $context, $registry, $resource, $resourceCollection, $data
        );
    }

    /**
     *
     * @param \Magento\Store\Model\Website $website
     * @return type
     */
    protected function _getOrdersCollection($website) {

        $order_age_in_minutes = $this->scopeConfig->getValue(self::ORDER_AGE_CONFIG_PATH, \Magento\Store\Model\ScopeInterface::SCOPE_WEBSITE, $website);
        $date = new \DateTime('@' . mktime(date("H"), date("i") - $order_age_in_minutes, 0, date("m"), date("d"), date("Y")));
        $collection = $this->salesResourceModelOrderCollectionFactory->create()->addAttributeToFilter('store_id', array('in' => $website->getStoreIds()))
                ->addAttributeToFilter('created_at', array('lteq' => $date->format('Y-m-d H:i:s')));

        $statuses_to_consider = explode(',', $this->scopeConfig->getValue(self::ORDER_STATUSES_CONFIG_PATH, \Magento\Store\Model\ScopeInterface::SCOPE_WEBSITE, $website));
        if (count($statuses_to_consider) > 0) {
            $collection->addAttributeToFilter('status', array('in' => $statuses_to_consider));
        } else {
            return array();
        }

        return $collection;
    }


    protected function _isAdyenHppMethodEnabled($order) {
        $isEnabled = FALSE;
        $enabled_adyenhpp_payment_methods = explode(",", $this->scopeConfig->getValue(self::ORDER_ADYENHPP_PAYMENT_METHODS_CONFIG_PATH, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $order->getStoreId()));
        if (count($enabled_adyenhpp_payment_methods) == 1 && $enabled_adyenhpp_payment_methods[0] == '') {
            $enabled_adyenhpp_payment_methods = array();
        }
        $additionalInformation = $order->getPayment()->getAdditionalInformation();
        if (!is_null($additionalInformation) && isset($additionalInformation['brand_code'])) {
            if (in_array($additionalInformation['brand_code'], $enabled_adyenhpp_payment_methods)) {
                $isEnabled = TRUE;
            }
        }

        return $isEnabled;
    }

    /**
     *
     * @param \Magento\Sales\Model\Order $order
     * @return boolean
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function _captureOnlinePayment($order) {

        try {

            //capture only enabled payment methods
            $enabled_payment_methods = explode(",", $this->scopeConfig->getValue(self::ORDER_PAYMENT_METHODS_CONFIG_PATH, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $order->getStoreId()));
            if (!in_array($order->getPayment()->getMethod(), $enabled_payment_methods)) {
                throw new \Magento\Framework\Exception\LocalizedException(__('cannot capture orders for payment code ' . $order->getPayment()->getMethod()));
            }

            //check the real payment method of Adyen HPP
            if ($order->getPayment()->getMethod() == self::ADYEN_HPP_PAYMENT_CODE) {
                if (in_array(self::ADYEN_HPP_PAYMENT_CODE, $enabled_payment_methods) && !$this->_isAdyenHppMethodEnabled($order)) {
                    throw new \Magento\Framework\Exception\LocalizedException(__('cannot capture orders paid with certain method of Adyen HPP'));
                }
            }

            //added in order to do not capture orders just voided
            if ($this->scopeConfig->getValue(self::VOIDED_ORDER_CHECK_CONFIG_PATH, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $order->getStoreId()) && $this->paymentscaptureHelper->isOrderVoided($order)) {
                throw new \Magento\Framework\Exception\LocalizedException(__('order transaction just voided or canceled'));
            }


            if (!$order->canInvoice()) {
                throw new \Magento\Framework\Exception\LocalizedException(__('cannot be invoiced'));
            }

            $invoice = $this->invoiceService->prepareInvoice($order);

            if (!$invoice->getTotalQty()) {
                throw new \Magento\Framework\Exception\LocalizedException(__('cannot create an invoice without products.'));
            }

            $invoice->setRequestedCaptureCase(\Magento\Sales\Model\Order\Invoice::CAPTURE_ONLINE);

            $invoice->register();
            $transactionSave = $this->transactionFactory->create()
                    ->addObject($invoice)
                    ->addObject($invoice->getOrder());
            $transactionSave->save();
        } catch (\Exception $e) {
            $this->customLogger->paymentsCaptureLog($order->getIncrementId() . ' ' . $e->getMessage());
            //$this->_sendFailedEmail($order->getIncrementId(), $e->getMessage());
            return false;
        }

        return true;
    }

    /**
     *
     * @param type $IncrementId
     * @param type $errorMsg
     * @return type
     * @todo make this work in M2
     */
    protected function _sendFailedEmail($IncrementId, $errorMsg) {

        if (!$this->scopeConfig->getValue('filoblu_paymentscapture/general/error_email_active', \Magento\Store\Model\ScopeInterface::SCOPE_STORE))
            return;

        $mail = Mage::getModel('core/email');
        //$mail->setToName('Your Name');
        $mail->setToEmail($this->scopeConfig->getValue('filoblu_paymentscapture/general/error_email', \Magento\Store\Model\ScopeInterface::SCOPE_STORE));
        $mail->setBody('Capture Error #' . $IncrementId . ' | Reason: ' . $errorMsg);
        $mail->setSubject('Capture Error ' . $IncrementId);
        $mail->setFromEmail($this->scopeConfig->getValue('trans_email/ident_custom1/email', \Magento\Store\Model\ScopeInterface::SCOPE_STORE));
        $mail->setFromName("Capture Error Notification");
        $mail->setType('text');
        try {
            $mail->send();
        } catch (Exception $e) {
            ;
        }
    }

    public function capture() {

        /** @var \Magento\Framework\App\ObjectManager $om */
        $om = \Magento\Framework\App\ObjectManager::getInstance();
        /** @var \Magento\Store\Model\StoreManagerInterface|\Magento\Store\Model\StoreManager $storeManager */
        $storeManager = $om->get('Magento\Store\Model\StoreManagerInterface');
        $websites = $storeManager->getWebsites();
        foreach ($websites as $website) {
            if (!$this->scopeConfig->isSetFlag(self::CAPTURE_ACTIVE_CONFIG_PATH, \Magento\Store\Model\ScopeInterface::SCOPE_WEBSITE, $website)) {
                continue;
            }
            $orders = $this->_getOrdersCollection($website);
            foreach ($orders as $order) {
                try {
                    $this->customLogger->paymentsCaptureLog('Processing order #' . $order->getIncrementId());
                    if ($this->_captureOnlinePayment($order)) {
                        if ($order->getStatus() != \Magento\Sales\Model\Order::STATE_PROCESSING) {
                            $order->setState(\Magento\Sales\Model\Order::STATE_PROCESSING, \Magento\Sales\Model\Order::STATE_PROCESSING, "Order processed by Payment capture cronJob")->save();
                        } else {
                            $order->addStatusHistoryComment("Order already processed", \Magento\Sales\Model\Order::STATE_PROCESSING);
                        }
                    } else {
                        $this->customLogger->paymentsCaptureLog('Skipped order #' . $order->getIncrementId());
                    }
                } catch (\Throwable $e) {
                    $this->customLogger->paymentsCaptureLog("Error processing order #{$order->getIncrementId()}: {$e->getMessage()}");
                    continue;
                }
            }
        }
    }

}
