<?php

namespace FiloBlu\PaymentMethodFilter\Observer;

use FiloBlu\PaymentMethodFilter\Helper\PaymentFilter;
use Magento\Framework\DataObject;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Framework\App\ObjectManager;

class RefiloFilter implements \Magento\Framework\Event\ObserverInterface
{
    /**
     * @var PaymentFilter
     */
    protected $helper;

    /**
     * @var CartRepositoryInterface
     */
    protected $quoteRepository;

    public function __construct(
        PaymentFilter $helper,
        CartRepositoryInterface $quoteRepository
    ){
        $this->helper = $helper;
        $this->quoteRepository = $quoteRepository;
    }
    public function execute(
        \Magento\Framework\Event\Observer $observer
    ) {
        /**
         * @var DataObject
         */
        $validPaymentMethods = $observer->getData('valid_payment_methods');
        // Get all valid payment methods
        $allPaymentDetails = $validPaymentMethods->getData('payment_details');
        // If filtered_payment exists means that some other observer has filtered payment methods
        $paymentDetails = is_null($validPaymentMethods->getData('filtered_payment')) ?
        $validPaymentMethods->getData('payment_details') : $validPaymentMethods->getData('filtered_payment');
        $cartId = $observer->getData('cart_id');
        $quote = $this->quoteRepository->getActive($cartId);

        $filteredValidPaymentMethods = [];
        $isFilterEnabled = $this->helper->isFilterEnabled();
        $isFilterCashOnDeliveryAndKlarnaWithGiftCardEnabled = $this->helper->filterCashOnDeliveryAndKlarnaWithGiftCard();
        $isRemoveCashOnDeliveryPayingByGiftCardEnabled = $this->helper->removeCashOnDeliveryPayingByGiftCard();
        $isFilterByShippingMethodEnabled = $this->helper->isFilterByShippingMethodEnabled();
        $paymentMethodWhitelist = $this->helper->paymentMethodWhitelist();
        $filterConditions = $this->helper->getFilterconditions();

        $quoteContainsSubscription = false;
        if($this->helper->existsSubscriptionsClass())
        {
            /**
             * @var \ParadoxLabs\Subscriptions\Model\Service\QuoteManager
             */
            $subscriptionClass = ObjectManager::getInstance()->get(\ParadoxLabs\Subscriptions\Model\Service\QuoteManager::class);
            $quoteContainsSubscription = $subscriptionClass->quoteContainsSubscription($quote);
        }

        foreach ($paymentDetails as $method_code) {
            if ($isFilterEnabled && !empty($paymentMethodWhitelist)) {
                if (empty($filterConditions)) {
                    if ($this->checkMethodAgainstWhitelist($method_code , $paymentMethodWhitelist)) {
                        $filteredValidPaymentMethods[$method_code] = $method_code;
                    }
                } else {
                    foreach ($filterConditions as $condition) {
                        switch ($condition) {
                            case 'giftcard':
                                if ($this->helper->quoteContainsGiftcard($quote) && $this->checkMethodAgainstWhitelist($method_code , $paymentMethodWhitelist)) {
                                    $filteredValidPaymentMethods[$method_code] = $method_code;
                                }
                                break;
                            case 'products': //Products in Cart
                                if ($this->helper->quoteContainsTriggerProducts($quote) && $this->checkMethodAgainstWhitelist($method_code , $paymentMethodWhitelist)) {
                                    $filteredValidPaymentMethods[$method_code] = $method_code;
                                }
                                break;
                            case 'product_custom_options': //Product Custom Options in Cart
                                if ($this->helper->quoteContainsTriggerCustomOptions($quote) && $this->checkMethodAgainstWhitelist($method_code , $paymentMethodWhitelist)) {
                                    $filteredValidPaymentMethods[$method_code] = $method_code;
                                }
                                break;
                            case 'product_attributes': //Product Attributes in Cart
                                if ($this->helper->quoteContainsTriggerAttributes($quote) && $this->checkMethodAgainstWhitelist($method_code , $paymentMethodWhitelist)) {
                                    $filteredValidPaymentMethods[$method_code] = $method_code;
                                }
                                break;
                            case 'product_subscriptions': //Products in subscriptions in Cart
                                if ($quoteContainsSubscription && $this->checkMethodAgainstWhitelist($method_code , $paymentMethodWhitelist)) {
                                    $filteredValidPaymentMethods[$method_code] = $method_code;
                                }
                                break;
                            case '*': //"any"
                                if ($this->checkMethodAgainstWhitelist($method_code , $paymentMethodWhitelist)) {
                                    $filteredValidPaymentMethods[$method_code] = $method_code;
                                }
                                break;
                            case '':
                            default:
                                break;
                        }
                    }
                }
            }
        }

        if(empty($filteredValidPaymentMethods))
        {
            $filteredValidPaymentMethods = array_combine($paymentDetails, $paymentDetails);
        }

        if($isRemoveCashOnDeliveryPayingByGiftCardEnabled) {
            if ($this->helper->isPayingWithGiftcard($quote)) {
                if(in_array($this->helper::CASH_ON_DELIVERY_VALUE, $filteredValidPaymentMethods)) {
                    unset($filteredValidPaymentMethods[$this->helper::CASH_ON_DELIVERY_VALUE]);
                }
            }
        }

        /**
         * If there is a gift card in the cart, in any case remove the cash on delivery and klarna from the payment methods list
         */
        if ($isFilterCashOnDeliveryAndKlarnaWithGiftCardEnabled && $this->helper->quoteContainsGiftcard($quote)) {
            foreach (PaymentFilter::FILTER_CASH_ON_DELIVERY_AND_KLARNA_WITH_GIFTCARD_CONFIG_VALUE as $methodToCheck) {
                if(in_array($methodToCheck, $filteredValidPaymentMethods)) {
                    unset($filteredValidPaymentMethods[$methodToCheck]);
                }
            }
        }

        foreach ($paymentDetails as $method_code) {

            if ($isFilterByShippingMethodEnabled) {
                $shippingMethod = $quote->getShippingAddress()->getShippingMethod();
                $filterShippingMethods = $this->helper->getFilterByShippingMethodMap();

                foreach($filterShippingMethods as $field) {
                    $shipping_code_filter = $field['shipping_code_filter'];

                    if ($shippingMethod == $shipping_code_filter &&
                        !$this->checkMethodAgainstWhitelist($method_code , $field['payment_code_filter'])) {
                        if(in_array($method_code, $filteredValidPaymentMethods)) {
                            unset($filteredValidPaymentMethods[$method_code]);
                        }
                    } /*
                        Removed because doesn't work with more shipping filter conditions
                        else {
                        if(!$isFilterEnabled && !$isRemoveCashOnDeliveryPayingByGiftCardEnabled) {
                            $filteredValidPaymentMethods[$method_code] = $method_code;
                        }
                    }*/
                }
            }
        }

        /**
         * Check if payment methods was filtered by other observers
         */
        if(empty($filteredValidPaymentMethods) && count($paymentDetails) != count($allPaymentDetails)){
            $filteredValidPaymentMethods = $paymentDetails;
        }
        $validPaymentMethods->setData('filtered_payment', $filteredValidPaymentMethods);
    }

    protected function checkMethodAgainstWhitelist($methodCode, $paymentMethodWhitelist): bool
    {
        return in_array($methodCode, $paymentMethodWhitelist);
    }
}
