<?php


namespace FiloBlu\PaymentMethodFilter\Helper;

use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Store\Model\ScopeInterface;
use FiloBlu\PaymentMethodFilter\Model\Serializer;
use Magento\Framework\App\ObjectManager;

class PaymentFilter extends AbstractHelper
{

    const FILTER_ENABLE_CONFIG_PATH = 'payment_method_filter/default/enable';
    const WHITELISTED_METHODS_CONFIG_PATH = 'payment_method_filter/default/whitelist_payment_methods';
    const FILTER_CONDITION_CONFIG_PATH = 'payment_method_filter/default/filter_condition';
    const PREVENT_BUY_GIFTCARD_WITH_GIFTCARD_CONFIG_PATH = 'payment_method_filter/gift_cards/prevent_paying_giftcards_with_giftcards';
    const REMOVE_CASH_ON_DELIVERY_PAYING_BY_GIFTCARD_CONFIG_PATH = 'payment_method_filter/gift_cards/remove_cash_on_delivery_if_paying_by_giftcard';
    const PRODUCT_IDS_CONFIG_PATH = 'payment_method_filter/default/product_ids';
    const PRODUCT_CUSTOM_OPTIONS_CONFIG_PATH = 'payment_method_filter/default/product_custom_options';
    const PRODUCT_CUSTOM_ATTRIBUTE_CONFIG_PATH = 'payment_method_filter/default/product_attributes';
    const LOGIN_AS_CUSTOMER_FILTER_ENABLE_CONFIG_PATH = 'payment_method_filter/login_as_customer/filter_login_as_customer';
    const LOGIN_AS_CUSTOMER_BLACKLISTED_METHODS_CONFIG_PATH = 'payment_method_filter/login_as_customer/blacklist_payment_methods';
    const SHIPPING_METHOD_FILTER_ENABLE_CONFIG_PATH = 'payment_method_filter/filter_carrier/enable_filter_carrier';
    const MAPPING_FILTER_BY_SHIPPING_PATH = 'payment_method_filter/filter_carrier/filter_carrier';
    const CASH_ON_DELIVERY_VALUE = 'msp_cashondelivery';
    const SUBSCRIPTIONS_CLASS = '\ParadoxLabs\Subscriptions\Model\Service\QuoteManager';
    const FILTER_CASH_ON_DELIVERY_AND_KLARNA_WITH_GIFTCARD_CONFIG_PATH = 'payment_method_filter/default/filter_cash_on_delivery_with_giftcard';
    const FILTER_CASH_ON_DELIVERY_AND_KLARNA_WITH_GIFTCARD_CONFIG_VALUE = ['checkmo', 'cashondelivery', 'msp_cashondelivery', 'klarna_kp', 'klarna_kco', 'klarna_pay_later', 'klarna_pay_over_time', 'klarna_direct_bank_transfer', 'klarna_direct_debit', 'klarna_pay_now'];

    private $serializer;

    /**
     * @param \Magento\Framework\App\Helper\Context $context
     */
    public function __construct(
        \Magento\Framework\App\Helper\Context $context,
        Serializer $serializer = null

    ) {
        parent::__construct($context);
        $this->serializer = $serializer ?: ObjectManager::getInstance()->get(Serializer::class);
    }

    public function isFilterEnabled() {
        return $this->scopeConfig->isSetFlag(self::FILTER_ENABLE_CONFIG_PATH, ScopeInterface::SCOPE_WEBSITE);
    }

    public function paymentMethodWhitelist() {
        $list = [];
        if ($this->scopeConfig->getValue(self::WHITELISTED_METHODS_CONFIG_PATH, ScopeInterface::SCOPE_WEBSITE) != '') {
            $list = explode(',', $this->scopeConfig->getValue(self::WHITELISTED_METHODS_CONFIG_PATH, ScopeInterface::SCOPE_WEBSITE));
        }
        return $list;
    }

    public function quoteContainsGiftcard($quote) {
        if (is_null($quote) == FALSE) {
            foreach ($quote->getAllItems() as $item) {
                if ($item->getProductType() == 'giftcard') {
                    return TRUE;
                }
            }
        }
        return FALSE;
    }

    public function getFilterconditions() {
        $conditions = [];
        $filterCondition = $this->scopeConfig->getValue(self::FILTER_CONDITION_CONFIG_PATH, ScopeInterface::SCOPE_WEBSITE);
        if (!is_null($filterCondition)) {
            $conditions = explode(',', $filterCondition);
        }
        return $conditions;
    }

    public function isGiftcardBuyingPrevented() {
        return $this->scopeConfig->isSetFlag(self::PREVENT_BUY_GIFTCARD_WITH_GIFTCARD_CONFIG_PATH, ScopeInterface::SCOPE_WEBSITE);
    }

    public function filterCashOnDeliveryAndKlarnaWithGiftCard() {
        return $this->scopeConfig->isSetFlag(self::FILTER_CASH_ON_DELIVERY_AND_KLARNA_WITH_GIFTCARD_CONFIG_PATH, ScopeInterface::SCOPE_WEBSITE);
    }

    public function removeCashOnDeliveryPayingByGiftCard() {
        return $this->scopeConfig->isSetFlag(self::REMOVE_CASH_ON_DELIVERY_PAYING_BY_GIFTCARD_CONFIG_PATH, ScopeInterface::SCOPE_WEBSITE);
    }

    /** @var \Magento\Quote\Model\Quote $quote */
    public function isPayingWithGiftcard($quote) {
        $giftcards = $quote->getGiftCards();
        if (is_null($giftcards)) {
            return FALSE;
        }
        return count($this->serializer->unserialize($giftcards)) > 0;
    }

    /** @var \Magento\Quote\Model\Quote $quote */
    public function quoteContainsTriggerProducts($quote) {
        $triggerProductIds = $this->scopeConfig->getValue(self::PRODUCT_IDS_CONFIG_PATH, ScopeInterface::SCOPE_WEBSITE);
        $triggerProductIds = array_map('trim', explode(',', $triggerProductIds));
        if (empty($triggerProductIds)) {
            return FALSE;
        }
        $productsInQuote = [];
        foreach($quote->getAllItems() as $quoteItem) {
            $productsInQuote[] = $quoteItem->getProductId();
        }
        return (count(array_intersect($productsInQuote, $triggerProductIds)) > 0);
    }

    public function quoteContainsTriggerCustomOptions($quote) {
        $triggerCustomOptionsLabels = $this->scopeConfig->getValue(self::PRODUCT_CUSTOM_OPTIONS_CONFIG_PATH, ScopeInterface::SCOPE_WEBSITE);
        $triggerCustomOptionsLabels = array_map('trim', explode(',', $triggerCustomOptionsLabels));
        if (empty($triggerCustomOptionsLabels)) {
            return FALSE;
        }
        /** @var @var \Magento\Quote\Model\Quote\Item $quoteItem */
        foreach($quote->getAllItems() as $quoteItem) {
            $product = $quoteItem->getProduct();
            if (!is_null($product)) {
                if (is_a($product->getCustomOption('option_ids'), 'Magento\Quote\Model\Quote\Item\Option')) {
                    $optionIds = $product->getCustomOption('option_ids');
                    foreach (explode(',', $optionIds->getValue()) as $optionId) {
                        $option = $product->getOptionById($optionId);
                        if (in_array($option->getDefaultTitle(), $triggerCustomOptionsLabels)) {
                            return TRUE;
                        }
                    }
                }
            }
        }
        return FALSE;
    }

    public function quoteContainsTriggerAttributes($quote) {
        $triggerCustomAttributeCodes = $this->scopeConfig->getValue(self::PRODUCT_CUSTOM_ATTRIBUTE_CONFIG_PATH, ScopeInterface::SCOPE_WEBSITE);
        $triggerCustomAttributeCodes = array_map('trim', explode(',', $triggerCustomAttributeCodes));
        if (empty($triggerCustomAttributeCodes)) {
            return FALSE;
        }
        /** @var @var \Magento\Quote\Model\Quote\Item $quoteItem */
        foreach($quote->getAllItems() as $quoteItem) {
            $product = $quoteItem->getProduct();
            if (!is_null($product)) {
                foreach ($triggerCustomAttributeCodes as $attributeCode) {
                    if (!is_null($product->getData($attributeCode)) && $product->getData($attributeCode) == '1') {
                        return TRUE;
                    }
                }
            }
        }
        return FALSE;
    }

    public function isLoginAsCustomerFilterEnabled() {
        return $this->scopeConfig->isSetFlag(self::LOGIN_AS_CUSTOMER_FILTER_ENABLE_CONFIG_PATH, ScopeInterface::SCOPE_WEBSITE);
    }

    public function loginAsCustomerPaymentMethodBlacklist() {
        $list = [];
        if ($this->scopeConfig->getValue(self::LOGIN_AS_CUSTOMER_BLACKLISTED_METHODS_CONFIG_PATH, ScopeInterface::SCOPE_WEBSITE) != '') {
            $list = explode(',', $this->scopeConfig->getValue(self::LOGIN_AS_CUSTOMER_BLACKLISTED_METHODS_CONFIG_PATH, ScopeInterface::SCOPE_WEBSITE));
        }
        return $list;
    }

    public function isAdminLoggedInAsCustomer($session) {
        return !is_null($session->getLoggedAsCustomerAdmindId());
    }

    public function isFilterByShippingMethodEnabled() {
        return $this->scopeConfig->isSetFlag(self::SHIPPING_METHOD_FILTER_ENABLE_CONFIG_PATH, ScopeInterface::SCOPE_WEBSITE);
    }

    public function getFilterByShippingMethodMap() {
        $shipping_filter_mapping = $this->scopeConfig->getValue(self::MAPPING_FILTER_BY_SHIPPING_PATH, ScopeInterface::SCOPE_WEBSITE);
        if ($shipping_filter_mapping) {
            return $this->serializer->unserialize($shipping_filter_mapping);
        } else {
            return [];
        }
    }

    public function existsSubscriptionsClass() {
        return class_exists(self::SUBSCRIPTIONS_CLASS);
    }
}
