<?php

namespace FiloBlu\OrderFlowRectifier\Observer;

use FiloBlu\OrderFlowRectifier\Helper\Data;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;

/**
 * Observer on Adyen notification to get cc_type value and update it in table sales_order_payment if missing
 * or fix it if payment is made by Klarna via Adyen
 * This observer needs the adyen module to be patched at least with adyen_notification_events_4303_update.patch
 */

class UpdateCCType implements ObserverInterface
{
    /**
     * @var ScopeConfigInterface
     */
    protected $config;

    public function __construct(
        ScopeConfigInterface $config
    )
    {
        $this->config = $config;
    }

    /**
     * @param Observer $observer
     */
    public function execute(Observer $observer)
    {
        if (in_array($observer->getEvent()->getNotificationCode(), ['CAPTURE', 'REFUND'])) {
            return true;
        }

        if (!$this->config->isSetFlag(Data::SCHEDULED_FIX_CCTYPE_ENABLED_CONFIG_PATH)) {
            return true;
        }

        $adyenPaymentMethod = $observer->getEvent()->getPaymentMethod();
        $order = $observer->getEvent()->getOrder();
        $orderPayment = $order->getPayment();
        $paymentMethod = $orderPayment->getMethod();
        
        if($paymentMethod == Data::PAYMENT_METHOD_ADYEN_CC)
        {
            if(array_key_exists($adyenPaymentMethod, Data::CC_TYPE_MAP))
            {
                $rightCcType = Data::CC_TYPE_MAP[$adyenPaymentMethod];

                if(is_null($orderPayment->getCcType()))
                {
                    // Fix cc_type if is null
                    $orderPayment->setCcType($rightCcType);
                }
                elseif($orderPayment->getCcType() != $rightCcType)
                {
                    // Fix cc_type if is wrong and different from adyen notification
                    $orderPayment->setCcType($rightCcType);
                    if(!is_null($orderPayment->getAdditionalInformation('brand_code'))) {
                        $additionalInformation = $orderPayment->getAdditionalInformation();
                        unset($additionalInformation['brand_code']);
                        $orderPayment->setAdditionalInformation($additionalInformation);
                    }
                }
            }
        }
        elseif ($paymentMethod == Data::PAYMENT_METHOD_ADYEN_PBL)
        {
            $ccType = $adyenPaymentMethod;

            if(array_key_exists($adyenPaymentMethod, Data::CC_TYPE_MAP))
            {
                $ccType = Data::CC_TYPE_MAP[$adyenPaymentMethod];
            }
            else
            {
                if(array_key_exists($adyenPaymentMethod, Data::CC_TYPE_HPP_MAP)) {
                    $ccType = Data::CC_TYPE_HPP_MAP[$adyenPaymentMethod];
                }

                if(is_null($orderPayment->getAdditionalInformation('brand_code'))) {
                    $additionalInformation = $orderPayment->getAdditionalInformation();
                    $additionalInformation['brand_code'] = $ccType;
                    $orderPayment->setAdditionalInformation($additionalInformation);
                }
            }

            $orderPayment->setCcType($ccType);
        }
        elseif ($paymentMethod == Data::PAYMENT_METHOD_ADYEN_HPP &&
            in_array($adyenPaymentMethod, [Data::KLARNA_CODE, Data::KLARNA_ACCOUNT_CODE]))
        {
            $ccTypeToSet = null;
            if($adyenPaymentMethod == Data::KLARNA_CODE)
            {
                $ccTypeToSet = Data::KLARNA_PAY_LATER_CODE;
            }
            elseif($adyenPaymentMethod == Data::KLARNA_ACCOUNT_CODE)
            {
                $countryId = $order->getBillingAddress()->getCountryId();
                if (isset(Data::COUNTRY_KLARNA_PAY_OVER_TIME_MAP[$countryId]))
                {
                    $ccTypeToSet = Data::COUNTRY_KLARNA_PAY_OVER_TIME_MAP[$countryId];
                }
            }
            $orderPayment->setCcType($ccTypeToSet);
        }

        if (is_null($orderPayment->getCcType())
            && ($adyenPaymentMethod == 'bcmc'
                || $paymentMethod == 'adyen_bcmc'
                || $orderPayment->getAdditionalInformation('brand_code') == 'bcmc')) {
            $orderPayment->setCcType(Data::CC_TYPE_MAP[$adyenPaymentMethod]);
        }

        return TRUE;
    }
}
