<?php

namespace FiloBlu\OrderFlowRectifier\Observer;

use FiloBlu\OrderFlowRectifier\Helper\AdyenBankTransferHelper;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\Serialize\SerializerInterface;

class ManageBankTransferData implements ObserverInterface
{
    /**
     * @var AdyenBankTransferHelper
     */
    protected $bankTransferHelper;
    /** @var SerializerInterface */
    private $serializer;

    /**
     * @param AdyenBankTransferHelper $bankTransferHelper
     * @param SerializerInterface $serializer
     */
    public function __construct(
        AdyenBankTransferHelper $bankTransferHelper,
        SerializerInterface     $serializer
    )
    {
        $this->bankTransferHelper = $bankTransferHelper;
        $this->serializer = $serializer;
    }

    public function execute(Observer $observer)
    {
        $order = $observer->getOrder();
        $payment = $order->getPayment();
        $method = $payment->getMethod();
        $ccType = $payment->getCcType();
        $bankTransferData = [];
        $historyComment = '';
        
        if ($method != 'adyen_bankTransfer_IBAN' && ($method != 'adyen_hpp' || $ccType != 'bankTransfer_IBAN')) {
            return true;
        }

        if (!$this->bankTransferHelper->automationBankTransferOrderIsEnabled()) {
            return true;
        }

        if ($this->bankTransferHelper->firstNotificationIsEnabled()) {
            $firstNotifyDate = $this->bankTransferHelper->calculateFirstNotifyDate();
            $bankTransferData[AdyenBankTransferHelper::FIRST_REMINDER_INDEX] = [
                AdyenBankTransferHelper::DATE_INDEX => $firstNotifyDate,
                AdyenBankTransferHelper::SENT_INDEX => 0
            ];
            $historyComment = 'First reminder on ' . $firstNotifyDate;
        }

        if ($this->bankTransferHelper->secondNotificationIsEnabled()) {
            $secondNotifyDate = $this->bankTransferHelper->calculateSecondNotifyDate();
            $bankTransferData[AdyenBankTransferHelper::SECOND_REMINDER_INDEX] = [
                AdyenBankTransferHelper::DATE_INDEX => $secondNotifyDate,
                AdyenBankTransferHelper::SENT_INDEX => 0
            ];
            if (!empty($historyComment)) {
                $historyComment .= ' \ ';
            }
            $historyComment .= 'Second reminder on ' . $secondNotifyDate;
        }

        if ($this->bankTransferHelper->automaticOrderCancellationIsEnabled()) {
            $cancellationDate = $this->bankTransferHelper->calculateOrderCancellationDate();
            $bankTransferData[AdyenBankTransferHelper::DELETE_ORDER_INDEX] = [
                AdyenBankTransferHelper::DATE_INDEX => $cancellationDate
            ];
            if (!empty($historyComment)) {
                $historyComment .= ' \ ';
            }
            $historyComment .= 'Order cancellation on ' . $cancellationDate;
        }

        if (!empty($bankTransferData)) {
            $payment->setAdditionalInformation(AdyenBankTransferHelper::NOTIFY_BANKTRANSFER_DATA, $this->serializer->serialize($bankTransferData));
            $order->addCommentToStatusHistory($historyComment, false);
        }

        return true;
    }
}
