<?php

namespace FiloBlu\OrderFlowRectifier\Model\Rectifiers;

use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\State;
use Magento\Sales\Model\Order\Email\Sender\OrderSender;
use Magento\Sales\Model\OrderRepository;

class SendEmailOrderConfirmationRectifier implements \FiloBlu\OrderFlowRectifier\Api\RectifierInterface
{

    const PREV_DAYS = 2;
    const ORDER_STATUS_TO_CHECK = ['confirmed','payment_confirmed','order_exported','payment_exported','sent_to_logistics','invoiced'];
    const PAYMENTS_DETAILS_TO_CHECK = ['ideal','directEbanking'];
    const PENDING_STATUS = 'pending';
    const MULTIBANCO_TYPE = 'multibanco';

    protected $connection;

    /**
     * @var OrderSender
     */
    protected $orderSender;

    /**
     * @var OrderRepository
     */
    protected $orderRepository;

    /**
     * @var State
     */
    protected $state;

    /**
     * @var ScopeConfigInterface
     */
    protected $globalConfig;

    public function __construct(
        \Magento\Framework\App\ResourceConnection $resource,
        OrderSender $orderSender,
        OrderRepository $orderRepository,
        State $state,
        ScopeConfigInterface $globalConfig
    )
    {
        $this->connection = $resource->getConnection();
        $this->orderSender = $orderSender;
        $this->orderRepository = $orderRepository;
        $this->state = $state;
        $this->globalConfig = $globalConfig;
    }

    public function getInfo() {
        return 'Management of the confirmation email for orders of the previous two days if not yet sent';
    }

    public function getTitle() {
        return 'Managing order confirmation email';
    }

    public function executeSingle($order) {

        /*
         * If the emails sending is asynchronous, only the 'send_email' field is updated
         */
        if ($this->globalConfig->getValue('sales_email/general/async_sending'))
        {
            $orderTable = $this->connection->getTableName('sales_order');
            $affectedRows = $this->connection->update(
                $orderTable,
                ['send_email' => 1],
                ['entity_id = ?' => $order->entity_id, 'increment_id = ?' => $order->increment_id]);
            $result = $affectedRows > 0;
        }
        else
        {
            try {
                $this->state->getAreaCode();
            } catch (\Exception $ex) {
                $this->state->setAreaCode('crontab');
            }

            $salesOrder = $this->orderRepository->get($order->entity_id);
            $result = $this->orderSender->send($salesOrder);
        }

        return $result;
    }

    public function loadOrders() {
        $orderTable = $this->connection->getTableName('sales_order');
        $orderPaymentTable = $this->connection->getTableName('sales_order_payment');
        $now = new \DateTime();
        $limitDate = $now->sub(new \DateInterval('P'.self::PREV_DAYS.'D'));

        $query = "SELECT so.entity_id, so.increment_id 
FROM {$orderTable} so
INNER JOIN {$orderPaymentTable} sop ON so.entity_id = sop.parent_id
WHERE so.email_sent IS NULL
AND so.send_email IS NULL
AND so.created_at >= '".$limitDate->format('Y-m-d')."'
AND (
(
sop.cc_type IN ('".implode("','",self::PAYMENTS_DETAILS_TO_CHECK)."') 
AND so.status IN ('".implode("','",self::ORDER_STATUS_TO_CHECK)."')
)
OR (
sop.cc_type = '".self::MULTIBANCO_TYPE."'
AND so.status = '".self::PENDING_STATUS."'
)
);";


        $ordersToBeRectified = $this->connection->fetchAll($query, [], \Zend_Db::FETCH_OBJ);
        return $ordersToBeRectified;
    }

}