<?php

namespace FiloBlu\OrderFlowRectifier\Model\Rectifiers;

use FiloBlu\OrderFlowRectifier\Api\RectifierInterface;
use Magento\Framework\App\State;
use Magento\Sales\Model\OrderRepository;

class KlarnaOrderRectifier implements RectifierInterface
{
    const ADYEN_NOTIFICATION_EVENT_CODE = 'AUTHORISATION';
    const PAYMENT_INFO_TO_CHECK = array(
        'pspReference' => 'pspReference',
        'adyen_avs_result' => 'avsResult',
        'adyen_cvc_result' => 'cvcResult',
        'adyen_refusal_reason_raw' => 'refusalReasonRaw',
        'adyen_acquirer_reference' => 'acquirerReference'
    );

    protected $connection;

    /**
     * @var OrderRepository
     */
    protected $orderRepository;

    /**
     * @var State
     */
    protected $state;

    public function __construct(
        \Magento\Framework\App\ResourceConnection $connection,
        OrderRepository $orderRepository,
        State $state
    )
    {
        $this->connection = $connection->getConnection();
        $this->orderRepository = $orderRepository;
        $this->state = $state;
    }

    public function getInfo()
    {
        return 'Adyen transaction data retrieval if missing and fix state for orders placed with Klarna';
    }

    public function getTitle()
    {
        return 'Rectifier for orders placed with Klarna';
    }

    public function executeSingle($order)
    {
        $orderTable = $this->connection->getTableName('sales_order');
        $orderPaymentTable = $this->connection->getTableName('sales_order_payment');
        $payment_additional_information = json_decode($order->additional_information, true);
        $payment_additional_information_is_json = true;
        if(json_last_error() !== JSON_ERROR_NONE)
        {
            $payment_additional_information = $order->additional_information ? unserialize($order->additional_information) : null;
            $payment_additional_information_is_json = false;
        }
        $adyen_additional_data = $order->additional_data ? unserialize($order->additional_data) : null;
        $orderPspReference = $order->pspreference;
        $dataToUpdate = array();

        if (is_null($order->cc_trans_id))
        {
            $dataToUpdate['cc_trans_id'] = $orderPspReference;
        }

        if (is_null($order->adyen_psp_reference))
        {
            $dataToUpdate['adyen_psp_reference'] = $orderPspReference;
        }

        $payAdditionalInfoUpdated = false;
        foreach (self::PAYMENT_INFO_TO_CHECK as $payInfo => $adyenData)
        {
            if(!array_key_exists($payInfo, $payment_additional_information))
            {
                if($payInfo == 'pspReference')
                {
                    $payment_additional_information[$payInfo] = $orderPspReference;
                    $payAdditionalInfoUpdated = true;
                }
                else
                {
                    $payment_additional_information[$payInfo] =
                        isset($adyen_additional_data[$adyenData]) ? $adyen_additional_data[$adyenData] : NULL;
                    $payAdditionalInfoUpdated = true;
                }
            }
        }

        if($payAdditionalInfoUpdated)
        {
            if($payment_additional_information_is_json)
            {
                $dataToUpdate['additional_information'] = json_encode($payment_additional_information, JSON_FORCE_OBJECT);
            }
            else
            {
                $dataToUpdate['additional_information'] = serialize($payment_additional_information);
            }
        }

        $affectedRows = 0;
        if(!empty($dataToUpdate))
        {
            $affectedRows = $this->connection->update(
                $orderPaymentTable,
                $dataToUpdate,
                ['entity_id = ?' => $order->payment_id, 'parent_id = ?' => $order->parent_id]);
        }

        $affectedRows2 = 0;
        if($order->status == 'order_exported' and $order->state == 'new')
        {
            $affectedRows2 = $this->connection->update(
                $orderTable,
                ['state' => 'processing'],
                ['entity_id = ?' => $order->entity_id, 'increment_id = ?' => $order->increment_id]);
        }
        elseif ($order->history_status == 'order_exported' and
            $order->status == 'pending' and $order->state == 'new')
        {
            $affectedRows2 = $this->connection->update(
                $orderTable,
                ['state' => 'processing', 'status' => 'order_exported'],
                ['entity_id = ?' => $order->entity_id, 'increment_id = ?' => $order->increment_id]);
        }
        elseif ($order->status == 'processing' and $order->state == 'processing')
        {
            try {
                $this->state->getAreaCode();
            } catch (\Exception $ex) {
                $this->state->setAreaCode('crontab');
            }

            $affectedRows2++;
            try {
                $salesOrder = $this->orderRepository->get($order->entity_id);
                $salesOrder->addStatusHistoryComment('Order processed by mod2-order-flow-rectifier, send capture from adyen');
                $salesOrder->save();
            } catch (\Exception $e) {
                $affectedRows2--;
            }
        }

        return ($affectedRows + $affectedRows2) > 0;
    }

    public function loadOrders()
    {
        $orderTable = $this->connection->getTableName('sales_order');
        $orderPaymentTable = $this->connection->getTableName('sales_order_payment');
        $adyenNotification = $this->connection->getTableName('adyen_notification');
        $orderHistoryTable = $this->connection->getTableName('sales_order_status_history');
        $now = new \DateTime();
        $limitDate = $now->sub(new \DateInterval('PT1H'));

        $query = "SELECT o.increment_id, o.entity_id, o.state, o.status, op.entity_id as payment_id,
                         op.parent_id, op.cc_trans_id, op.additional_information, op.adyen_psp_reference,
                         oh.status as history_status, an.entity_id as notification_id, an.pspreference, an.additional_data
FROM {$orderTable} o
INNER JOIN {$orderPaymentTable} op ON o.entity_id = op.parent_id
AND op.method IN ('adyen_hpp') AND op.cc_type IN ('klarna', 'klarna_account', 'klarna_paynow')
INNER JOIN {$adyenNotification} an ON o.increment_id = an.merchant_reference
JOIN (
    SELECT b1.* FROM {$orderHistoryTable} b1
    JOIN
    (
      SELECT parent_id, MAX(entity_id) As maxDate
      FROM {$orderHistoryTable}
      GROUP BY parent_id
    ) b2
    ON b1.parent_id = b2.parent_id
    AND b1.entity_id = b2.maxDate
) oh
ON o.entity_id = oh.parent_id
WHERE
      (
          (o.status in ('order_exported', 'pending', 'processing') AND (
          op.adyen_psp_reference IS NULL OR
          op.cc_trans_id IS NULL OR
          op.additional_information NOT LIKE '%pspReference%'))
          OR
          (o.status = 'pending' AND
           o.state = 'new' AND
           oh.status = 'order_exported')
      ) AND
      o.created_at <= '".$limitDate->format('Y-m-d H:i:s')."' AND
      an.event_code = '".self::ADYEN_NOTIFICATION_EVENT_CODE."' AND
      an.success = 'true'
ORDER BY o.entity_id DESC";

        $ordersToBeRectified = $this->connection->fetchAll($query, [], \Zend_Db::FETCH_OBJ);
        return $ordersToBeRectified;
    }
}
