<?php

namespace FiloBlu\OrderFlowRectifier\Model\Rectifiers;

use FiloBlu\OrderFlowRectifier\Api\RectifierInterface;
use Magento\Framework\App\State;
use Magento\Sales\Model\OrderRepository;

class ClearPayPalExpressOrderRectifier implements RectifierInterface
{
    protected $connection;

    /**
     * @var OrderRepository
     */
    protected $orderRepository;

    /**
     * @var State
     */
    protected $state;

    public function __construct(
        \Magento\Framework\App\ResourceConnection $connection,
        OrderRepository $orderRepository,
        State $state
    )
    {
        $this->connection = $connection->getConnection();
        $this->orderRepository = $orderRepository;
        $this->state = $state;
    }

    public function getInfo()
    {
        return 'Cancel orders made with PayPal Express Checkout still in processing and without payment';
    }

    public function getTitle()
    {
        return 'Rectifier for clear PayPal Express Checkout orders';
    }

    public function executeSingle($order)
    {
        try {
            $this->state->getAreaCode();
        } catch (\Exception $ex) {
            $this->state->setAreaCode('crontab');
        }

        $changeStatusCounter = $order->count_status;
        $affectedRows = 0;

        // If are not presents status changes (so there is only one row) it means that status has never changed from "Processing", so I can cancel the order.
        if($changeStatusCounter == 1)
        {
            $affectedRows = 1;
            try {
                $salesOrder = $this->orderRepository->get($order->entity_id);
                $salesOrder->cancel();
                $salesOrder->addStatusHistoryComment('Order payment not completed, processed by mod2-order-flow-rectifier.');
                $salesOrder->save();
            } catch (\Exception $e) {
                $affectedRows--;
            }
        }

        return $affectedRows;
    }

    public function loadOrders()
    {
        $orderTable = $this->connection->getTableName('sales_order');
        $orderHistoryTable = $this->connection->getTableName('sales_order_status_history');
        $orderPaymentTable = $this->connection->getTableName('sales_order_payment');
        $now = new \DateTime();
        $limitDate = $now->sub(new \DateInterval('PT4H'));

        $query = "SELECT o.increment_id, o.entity_id, COUNT(oh.entity_id) as count_status
FROM {$orderTable} o
INNER JOIN {$orderHistoryTable} oh ON o.entity_id = oh.parent_id
INNER JOIN {$orderPaymentTable} op ON o.entity_id = op.parent_id
AND op.method IN ('paypal_express')
WHERE
      o.status = 'processing' AND
      oh.created_at <= '".$limitDate->format('Y-m-d H:i:s')."'
GROUP BY oh.parent_id
ORDER BY o.entity_id DESC";

        $ordersToBeRectified = $this->connection->fetchAll($query, [], \Zend_Db::FETCH_OBJ);

        return $ordersToBeRectified;
    }
}