<?php

namespace FiloBlu\OrderFlowRectifier\Model\Order\Email\Sender;

use FiloBlu\OrderFlowRectifier\Helper\AdyenBankTransferHelper;
use Magento\Framework\App\Area;
use Magento\Framework\App\AreaList;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\App\State;
use Magento\Framework\Event\ManagerInterface;
use Magento\Framework\Serialize\SerializerInterface;
use Magento\Framework\Url;
use Magento\Payment\Helper\Data as PaymentHelper;
use Magento\Sales\Model\Order;
use Magento\Sales\Model\Order\Address\Renderer;
use Magento\Sales\Model\Order\Email\Container\OrderIdentity;
use Magento\Sales\Model\Order\Email\Container\Template;
use Magento\Sales\Model\ResourceModel\Order as OrderResource;
use Magento\Store\Model\ScopeInterface;

class OrderSender extends \Magento\Sales\Model\Order\Email\Sender\OrderSender
{
    const XML_PATH_OFFLINE_PAYMENT_DETAILS_TEMPLATE = 'order_flow_rectifier/adyen_email_configurations/offline_payment_details_template';
    const ADYEN_EMAIL_CONFIGURATIONS_CONFIG_PATH = 'order_flow_rectifier/adyen_email_configurations/adyen_email_configurations_enabled';
    const PRIVACY_POLICY_CONFIG_PATH = 'order_flow_rectifier/adyen_email_configurations/privacy_policy_path';

    /**
     * @var ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * @var ResourceConnection
     */
    protected $connection;
    /** @var State */
    protected $state;
    /** @var SerializerInterface */
    private $serializer;
    /** @var Url */
    private $url;
    /** @var AreaList */
    private $areaList;

    /**
     * @param Template $templateContainer
     * @param OrderIdentity $identityContainer
     * @param Order\Email\SenderBuilderFactory $senderBuilderFactory
     * @param \Psr\Log\LoggerInterface $logger
     * @param Renderer $addressRenderer
     * @param PaymentHelper $paymentHelper
     * @param OrderResource $orderResource
     * @param ScopeConfigInterface $globalConfig
     * @param ManagerInterface $eventManager
     * @param ScopeConfigInterface $scopeConfig
     * @param ResourceConnection $connection
     * @param SerializerInterface $serializer
     * @param Url $url
     * @param State $state
     * @param AreaList $areaList
     */
    public function __construct(
        Template                                              $templateContainer,
        OrderIdentity                                         $identityContainer,
        \Magento\Sales\Model\Order\Email\SenderBuilderFactory $senderBuilderFactory,
        \Psr\Log\LoggerInterface                              $logger,
        Renderer                                              $addressRenderer,
        PaymentHelper                                         $paymentHelper,
        OrderResource                                         $orderResource,
        \Magento\Framework\App\Config\ScopeConfigInterface    $globalConfig,
        ManagerInterface                                      $eventManager,
        ScopeConfigInterface                                  $scopeConfig,
        ResourceConnection                                    $connection,
        SerializerInterface                                   $serializer,
        Url                                                   $url,
        State                                                 $state,
        AreaList                                              $areaList
    )
    {
        parent::__construct($templateContainer, $identityContainer, $senderBuilderFactory, $logger, $addressRenderer, $paymentHelper, $orderResource, $globalConfig, $eventManager);
        $this->scopeConfig = $scopeConfig;
        $this->connection = $connection->getConnection();
        $this->serializer = $serializer;
        $this->url = $url;
        $this->state = $state;
        $this->areaList = $areaList;
    }

    /**
     * @param Order $order
     * @param bool $forceSyncMode
     * @return bool
     */
    public function send(Order $order, $forceSyncMode = false)
    {
        $isOrderToPayOffline = $this->isOrderToPayOffline($order);
        if ($isOrderToPayOffline) {

            if (!$order->getOfflinePaymentEmailSent()) {
                if ($this->state->getAreaCode() == Area::AREA_CRONTAB) {
                    $area = $this->areaList->getArea($this->state->getAreaCode());
                    $area->load(Area::PART_TRANSLATE);
                }
                $this->identityContainer->setStore($order->getStore());
                $order->setSendEmail($this->identityContainer->isEnabled());
                if ($this->checkAndSend($order)) {
                    if ($this->globalConfig->getValue('sales_email/general/async_sending')) {
                        $order->setEmailSent(null);
                    } else {
                        $order->setEmailSent(false);
                    }
                    $order->setData('offline_payment_email_sent', 1);
                    $this->orderResource->saveAttribute($order, ['email_sent', 'offline_payment_email_sent']);
                }

                $this->orderResource->saveAttribute($order, 'send_email');
            }

            return false;
        } else {
            return parent::send($order, $forceSyncMode);
        }
    }

    /**
     * @param Order $order
     * @return bool
     */
    protected function isOrderToPayOffline(Order $order)
    {
        $isOfflinePaymentMethod = false;
        $paymentRecived = false;

        if (!$this->scopeConfig->isSetFlag(self::ADYEN_EMAIL_CONFIGURATIONS_CONFIG_PATH, ScopeInterface::SCOPE_STORE, $order->getStoreId())) {
            return $isOfflinePaymentMethod;
        }

        try {
            $payment = $order->getPayment();
            $paymentMethod = $payment->getMethod();
            $ccType = $payment->getCcType();
        } catch (\Exception $e) {
            $this->logger->error('Payment missing for order '.$order->getIncrementId());
            $this->logger->error($e->getMessage());
            return false;
        }

        if ($paymentMethod == 'adyen_pay_by_link'
            || ($paymentMethod == 'adyen_hpp' && in_array($ccType, ['bankTransfer_IBAN', 'multibanco']))
            || (in_array($paymentMethod, ['adyen_bankTransfer_IBAN', 'adyen_multibanco']))) {
            $isOfflinePaymentMethod = true;
            $adyenNotification = $this->connection->getTableName('adyen_notification');
            $query = "SELECT pspreference FROM {$adyenNotification}
                    WHERE merchant_reference = '" . $order->getIncrementId() . "'
                    AND event_code != 'PENDING'
                    AND success = 'true'";

            $notification = $this->connection->fetchAll($query, [], \Zend_Db::FETCH_OBJ);
            if (count($notification) > 0) {
                $paymentRecived = true;
            }
        }

        return $isOfflinePaymentMethod && !$paymentRecived;
    }

    /**
     * @param Order $order
     * @param $forceSyncMode
     * @return bool
     */
    public function sendReminder(Order $order, $forceSyncMode = false)
    {
        $this->identityContainer->setStore($order->getStore());
        if ($this->checkAndSend($order)) {
            return true;
        }
        return false;
    }

    /**
     * @param Order $order
     * @return void
     */
    protected function prepareTemplate(Order $order)
    {
        parent::prepareTemplate($order);
        $payment = $order->getPayment();
        if (!is_null($payment->getAdditionalInformation(AdyenBankTransferHelper::FIRST_REMINDER_INDEX)) ||
            !is_null($payment->getAdditionalInformation(AdyenBankTransferHelper::SECOND_REMINDER_INDEX))) {
            $isFirstReminder = false;
            $isSecondReminder = false;
            $deleteDate = '';

            if (!is_null($payment->getAdditionalInformation(AdyenBankTransferHelper::NOTIFY_BANKTRANSFER_DATA)) &&
                $order->canCancel()) {
                $bankTransferData = $this->serializer->unserialize($payment->getAdditionalInformation(AdyenBankTransferHelper::NOTIFY_BANKTRANSFER_DATA));
                if (array_key_exists(AdyenBankTransferHelper::DELETE_ORDER_INDEX, $bankTransferData)) {
                    $date = new \DateTime($bankTransferData[AdyenBankTransferHelper::DELETE_ORDER_INDEX][AdyenBankTransferHelper::DATE_INDEX]);
                    /* In the email I write the previous day as the limit for order cancellation */
                    $date = $date->sub(new \DateInterval('P1D'));
                    $now = new \DateTime();
                    if ($date >= $now) {
                        $deleteDate = $date->format('d-m-Y');
                    }
                }
            }

            $templateVars = $this->templateContainer->getTemplateVars();
            $additonalInfo = $payment->getAdditionalInformation();
            $templateVars['order']['deleteDate'] = $deleteDate;
            $templateVars['order']['amount'] = $order->getOrderCurrency()->formatPrecision($order->getGrandTotal(), 2, [], false, false);
            $templateVars['order']['reference'] = $additonalInfo['action']['reference'];
            $templateVars['order']['iban'] = $additonalInfo['action']['iban'];
            $templateVars['order']['bic'] = $additonalInfo['action']['bic'];
            $templateVars['order']['beneficiary'] = $additonalInfo['action']['beneficiary'];
            $this->templateContainer->setTemplateVars($templateVars);

            $templateId = $this->scopeConfig->getValue(
                AdyenBankTransferHelper::BANKTRANSFER_REMAINDER_EMAIL_TEMPLATE_PATH,
                \Magento\Store\Model\ScopeInterface::SCOPE_STORE,
                $order->getStoreId()
            );
            $this->templateContainer->setTemplateId($templateId);
        } else {
            $isOrderToPayOffline = $this->isOrderToPayOffline($order);

            if ($isOrderToPayOffline) {
                $isPayByLink = false;
                $isBankTransfer = false;
                $privacyPolicyUrl = '';
                if ($payment->getMethod() == 'adyen_pay_by_link') {
                    $privacyPolicyPath = $this->scopeConfig->getValue(
                        self::PRIVACY_POLICY_CONFIG_PATH,
                        \Magento\Store\Model\ScopeInterface::SCOPE_STORE,
                        $order->getStoreId()
                    );
                    $privacyPolicyUrl = $this->url->setScope($order->getStore())->getUrl('') . $privacyPolicyPath . '/';
                    $isPayByLink = true;
                } else {
                    if ($payment->getMethod() == 'adyen_bankTransfer_IBAN' ||
                        $payment->getCcType() == 'bankTransfer_IBAN') {
                        $isBankTransfer = true;
                    }
                }
                $templateVars = $this->templateContainer->getTemplateVars();
                $templateVars['order']['isPayByLink'] = $isPayByLink;
                $templateVars['order']['isBankTransfer'] = $isBankTransfer;
                $templateVars['order']['privacyPolicyUrl'] = $privacyPolicyUrl;
                $this->templateContainer->setTemplateVars($templateVars);

                $templateId = $this->scopeConfig->getValue(
                    self::XML_PATH_OFFLINE_PAYMENT_DETAILS_TEMPLATE,
                    \Magento\Store\Model\ScopeInterface::SCOPE_STORE,
                    $order->getStoreId()
                );
                $this->templateContainer->setTemplateId($templateId);
            }
        }
    }
}
