<?php

namespace FiloBlu\OrderFlowRectifier\Model\Rectifiers;

use FiloBlu\OrderFlowRectifier\Api\RectifierInterface;
use Magento\Framework\App\State;
use Magento\Sales\Model\OrderRepository;

class OrderAdyenHppPaymentDataIncompleteRectifier implements RectifierInterface
{
    const ADYEN_NOTIFICATION_EVENT_CODE = 'AUTHORISATION';
    const ORDER_STATUS_TO_CHECK = "'order_exported', 'pending', 'processing', 'complete', 'sent_to_logistics', 'payment_confirmed', 'payment_exported', 'shipped'";
    const PAYMENT_INFO_TO_CHECK = array(
        'pspReference' => 'pspReference',
        'adyen_avs_result' => 'avsResult',
        'adyen_cvc_result' => 'cvcResult',
        'adyen_refusal_reason_raw' => 'refusalReasonRaw',
        'adyen_acquirer_reference' => 'acquirerReference'
    );

    protected $connection;

    /**
     * @var OrderRepository
     */
    protected $orderRepository;

    /**
     * @var State
     */
    protected $state;

    public function __construct(
        \Magento\Framework\App\ResourceConnection $connection,
        OrderRepository $orderRepository,
        State $state
    )
    {
        $this->connection = $connection->getConnection();
        $this->orderRepository = $orderRepository;
        $this->state = $state;
    }

    public function getInfo()
    {
        return 'Adyen transaction data retrieval for adyen_hpp orders with data missing';
    }

    public function getTitle()
    {
        return 'Retrieve adyen_hpp orders transaction info if missing';
    }

    public function executeSingle($order)
    {
        $orderPaymentTable = $this->connection->getTableName('sales_order_payment');
        $payment_additional_information = json_decode($order->additional_information, true);
        $payment_additional_information_is_json = true;
        if(json_last_error() !== JSON_ERROR_NONE)
        {
            $payment_additional_information = $order->additional_information ? unserialize($order->additional_information) : null;
            $payment_additional_information_is_json = false;
        }
        $adyen_additional_data = $order->additional_data ? unserialize($order->additional_data) : null;
        $orderPspReference = $order->pspreference;
        $dataToUpdate = array();

        if (is_null($order->cc_trans_id))
        {
            $dataToUpdate['cc_trans_id'] = $orderPspReference;
        }

        if (is_null($order->adyen_psp_reference))
        {
            $dataToUpdate['adyen_psp_reference'] = $orderPspReference;
        }

        $payAdditionalInfoUpdated = false;
        foreach (self::PAYMENT_INFO_TO_CHECK as $payInfo => $adyenData)
        {
            if(!array_key_exists($payInfo, $payment_additional_information))
            {
                if($payInfo == 'pspReference')
                {
                    $payment_additional_information[$payInfo] = $orderPspReference;
                    $payAdditionalInfoUpdated = true;
                }
                else
                {
                    $payment_additional_information[$payInfo] =
                        isset($adyen_additional_data[$adyenData]) ? $adyen_additional_data[$adyenData] : NULL;
                    $payAdditionalInfoUpdated = true;
                }
            }
        }

        if($payAdditionalInfoUpdated)
        {
            if($payment_additional_information_is_json)
            {
                $dataToUpdate['additional_information'] = json_encode($payment_additional_information, JSON_FORCE_OBJECT);
            }
            else
            {
                $dataToUpdate['additional_information'] = serialize($payment_additional_information);
            }
        }

        $affectedRows = 0;
        if(!empty($dataToUpdate))
        {
            $affectedRows = $this->connection->update(
                $orderPaymentTable,
                $dataToUpdate,
                ['entity_id = ?' => $order->payment_id, 'parent_id = ?' => $order->parent_id]);
        }

        return $affectedRows > 0;
    }

    public function loadOrders()
    {
        /* Rectifier obsolete for adyen 9 */
        $orderTable = $this->connection->getTableName('sales_order');
        $orderPaymentTable = $this->connection->getTableName('sales_order_payment');
        $adyenNotification = $this->connection->getTableName('adyen_notification');
        $now = new \DateTime();
        $limitDate = $now->sub(new \DateInterval('PT1H'));
        $dateMin = new \DateTime('2020-01-01');;

        $query = "SELECT o.increment_id, o.entity_id, o.state, o.status, op.entity_id as payment_id,
                         op.parent_id, op.cc_trans_id, op.additional_information, op.adyen_psp_reference,
                         an.entity_id as notification_id, an.pspreference, an.additional_data
FROM {$orderTable} o
INNER JOIN {$orderPaymentTable} op ON o.entity_id = op.parent_id
AND op.method IN ('adyen_hpp') AND op.cc_type IN ('ideal', 'directEbanking')
INNER JOIN {$adyenNotification} an ON o.increment_id = an.merchant_reference
WHERE
      (
          op.adyen_psp_reference IS NULL OR
          op.cc_trans_id IS NULL OR
          op.additional_information NOT LIKE '%pspReference%'
      ) AND
      o.status in (".self::ORDER_STATUS_TO_CHECK.") AND
      o.created_at <= '".$limitDate->format('Y-m-d H:i:s')."' AND
      o.created_at > '".$dateMin->format('Y-m-d H:i:s')."' AND
      an.event_code = '".self::ADYEN_NOTIFICATION_EVENT_CODE."' AND
      an.success = 'true'
ORDER BY o.entity_id DESC";

        $ordersToBeRectified = $this->connection->fetchAll($query, [], \Zend_Db::FETCH_OBJ);
        return $ordersToBeRectified;
    }
}
