<?php

namespace FiloBlu\OrderFlowRectifier\Model\Rectifiers;

use FiloBlu\OrderFlowRectifier\Api\RectifierInterface;
use FiloBlu\OrderFlowRectifier\Helper\Data;
use Magento\Framework\App\State;
use Magento\Sales\Model\OrderRepository;

class ClearKlarnaOrderRectifier implements RectifierInterface
{
    protected $connection;

    /**
     * @var OrderRepository
     */
    protected $orderRepository;

    /**
     * @var State
     */
    protected $state;

    public function __construct(
        \Magento\Framework\App\ResourceConnection $connection,
        OrderRepository $orderRepository,
        State $state
    )
    {
        $this->connection = $connection->getConnection();
        $this->orderRepository = $orderRepository;
        $this->state = $state;
    }

    public function getInfo()
    {
        return 'Cancel orders made with Klarna still in pendig and without adyen notification';
    }

    public function getTitle()
    {
        return 'Rectifier for clear Klarna orders';
    }

    public function executeSingle($order)
    {
        try {
            $this->state->getAreaCode();
        } catch (\Exception $ex) {
            $this->state->setAreaCode('crontab');
        }

        $affectedRows = 1;
        try {
            $salesOrder = $this->orderRepository->get($order->entity_id);
            $payment = $salesOrder->getPayment();
            $ccType = $payment->getCcType();
            if(in_array($ccType, [Data::KLARNA_CODE, Data::KLARNA_ACCOUNT_CODE])) {

                $ccTypeToSet = null;
                if($ccType == Data::KLARNA_CODE) {
                    $ccTypeToSet = Data::KLARNA_PAY_LATER_CODE;
                } elseif ($ccType == Data::KLARNA_ACCOUNT_CODE) {
                    $countryId = $salesOrder->getBillingAddress()->getCountryId();
                    if(isset(Data::COUNTRY_KLARNA_PAY_OVER_TIME_MAP[$countryId])) {
                        $ccTypeToSet = Data::COUNTRY_KLARNA_PAY_OVER_TIME_MAP[$countryId];
                    }
                }
                $payment->setCcType($ccTypeToSet);
            }
            $salesOrder->cancel();
            $salesOrder->addStatusHistoryComment('Order without payment and adyen notification, processed by mod2-order-flow-rectifier.');
            $salesOrder->save();
        } catch (\Exception $e) {
            $affectedRows--;
        }

        return $affectedRows;
    }

    public function loadOrders()
    {
        $orderTable = $this->connection->getTableName('sales_order');
        $orderPaymentTable = $this->connection->getTableName('sales_order_payment');
        $adyenNotification = $this->connection->getTableName('adyen_notification');
        $now = new \DateTime();
        $limitDate = $now->sub(new \DateInterval('PT13H'));

        $query = "SELECT o.increment_id, o.entity_id
FROM {$orderTable} o
INNER JOIN {$orderPaymentTable} op ON o.entity_id = op.parent_id
AND op.method IN ('adyen_hpp') AND op.cc_type IN ('klarna', 'klarna_account', 'klarna_paynow')
WHERE
      NOT EXISTS (
          SELECT * FROM {$adyenNotification} an
          WHERE an.merchant_reference = o.increment_id
      ) AND
      o.status = 'pending' AND
      o.created_at <= '".$limitDate->format('Y-m-d H:i:s')."'
ORDER BY o.entity_id DESC";

        $ordersToBeRectified = $this->connection->fetchAll($query, [], \Zend_Db::FETCH_OBJ);

        return $ordersToBeRectified;
    }
}