<?php

namespace FiloBlu\OrderFlowRectifier\Model;

use Magento\Framework\App\Config\ValueFactory;
use Magento\Framework\App\ObjectManager;
use Magento\Sales\Model\Order\Email\Container\IdentityInterface;
use Magento\Sales\Model\ResourceModel\Collection\AbstractCollection;
use Magento\Store\Model\StoreManagerInterface;

class EmailSenderHandler extends \Magento\Sales\Model\EmailSenderHandler
{
    const EXCLUDE_CANCELED_ORDERS_FROM_COLLECTION_CONFIG_PATH = 'order_flow_rectifier/adyen_email_configurations/exclude_canceled_orders_from_collection';
    const DAYS_TO_SEND_ORDERS_EMAILS_CONFIG_PATH = 'order_flow_rectifier/adyen_email_configurations/days_to_send_orders_emails';

    /**
     * @var IdentityInterface
     */
    private $identityContainer;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @param \Magento\Sales\Model\Order\Email\Sender $emailSender
     * @param \Magento\Sales\Model\ResourceModel\EntityAbstract $entityResource
     * @param AbstractCollection $entityCollection
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $globalConfig
     * @param IdentityInterface|null $identityContainer
     * @param StoreManagerInterface|null $storeManager
     * @param ValueFactory|null $configValueFactory
     * @param string|null $modifyStartFromDate
     */
    public function __construct(
        \Magento\Sales\Model\Order\Email\Sender $emailSender,
        \Magento\Sales\Model\ResourceModel\EntityAbstract $entityResource,
        AbstractCollection $entityCollection,
        \Magento\Framework\App\Config\ScopeConfigInterface $globalConfig,
        IdentityInterface $identityContainer = null,
        StoreManagerInterface $storeManager = null,
        ?ValueFactory $configValueFactory = null,
        ?string $modifyStartFromDate = null
    ) {
        parent::__construct($emailSender, $entityResource, $entityCollection, $globalConfig, $identityContainer, $storeManager, $configValueFactory, $modifyStartFromDate);

        $this->identityContainer = $identityContainer ?: ObjectManager::getInstance()
            ->get(\Magento\Sales\Model\Order\Email\Container\NullIdentity::class);
        $this->storeManager = $storeManager ?: ObjectManager::getInstance()
            ->get(StoreManagerInterface::class);
    }

    public function sendEmails()
    {
        if ($this->globalConfig->isSetFlag(self::EXCLUDE_CANCELED_ORDERS_FROM_COLLECTION_CONFIG_PATH)) {
            if ($this->globalConfig->getValue('sales_email/general/async_sending')) {
                $this->entityCollection->addFieldToFilter('status', ['neq' => 'canceled']);
                $this->entityCollection->addFieldToFilter('send_email', ['eq' => 1]);
                $this->entityCollection->addFieldToFilter('email_sent', ['null' => true]);
                $daysAmount = $this->globalConfig->getValue(self::DAYS_TO_SEND_ORDERS_EMAILS_CONFIG_PATH);
                $startFromDate = new \DateTime();
                $startFromDate->sub(new \DateInterval('P'.$daysAmount.'D'));
                $this->entityCollection->addFieldToFilter('created_at', ['from' => $startFromDate->format('Y-m-d H:i:s')]);
                $this->entityCollection->setPageSize(
                    $this->globalConfig->getValue('sales_email/general/sending_limit')
                );

                /** @var \Magento\Store\Api\Data\StoreInterface[] $stores */
                $stores = $this->getStores(clone $this->entityCollection);

                /** @var \Magento\Store\Model\Store $store */
                foreach ($stores as $store) {
                    $this->identityContainer->setStore($store);
                    if (!$this->identityContainer->isEnabled()) {
                        continue;
                    }
                    $entityCollection = clone $this->entityCollection;
                    $entityCollection->addFieldToFilter('store_id', $store->getId());

                    /** @var \Magento\Sales\Model\AbstractModel $item */
                    foreach ($entityCollection->getItems() as $item) {
                        if ($this->emailSender->send($item, true)) {
                            $this->entityResource->saveAttribute(
                                $item->setEmailSent(true),
                                'email_sent'
                            );
                        }
                    }
                }
            }
        } else {
            parent::sendEmails();
        }
    }

    /**
     * Get stores for given entities.
     *
     * @param ResourceModel\Collection\AbstractCollection $entityCollection
     * @return \Magento\Store\Api\Data\StoreInterface[]
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    private function getStores(
        AbstractCollection $entityCollection
    ): array {
        $stores = [];

        $entityCollection->addAttributeToSelect('store_id')->getSelect()->group('store_id');
        /** @var \Magento\Sales\Model\EntityInterface $item */
        foreach ($entityCollection->getItems() as $item) {
            /** @var \Magento\Store\Model\StoreManagerInterface $store */
            $store = $this->storeManager->getStore($item->getStoreId());
            $stores[$item->getStoreId()] = $store;
        }

        return $stores;
    }
}
