<?php

namespace FiloBlu\OrderFlowRectifier\Model;

use DateInterval;
use DateTime;
use Magento\Email\Model\BackendTemplate;
use Magento\Framework\App\Area;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Data\Collection\AbstractDb;
use Magento\Framework\Mail\Template\TransportBuilder;
use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Model\Context;
use Magento\Framework\Model\ResourceModel\AbstractResource;
use Magento\Framework\Registry;
use Magento\Store\Model\Store;
use Zend_Db;

class CheckAdyenAuthorization extends AbstractModel
{
    const CHECK_ADYEN_AUTHORIZATION_ENABLED_CONFIG_PATH = 'order_flow_rectifier/check_adyen_authorization/check_adyen_authorization_enable';
    const PAST_DAYS_TO_CHECK_TRANSACTIONS_CONFIG_PATH = 'order_flow_rectifier/check_adyen_authorization/past_days_to_check_transactions';
    const NUMBER_DAYS_TO_SEND_NOTIFICATION_CONFIG_PATH = 'order_flow_rectifier/check_adyen_authorization/number_days_to_send_notification';
    const METHODS_WITHOUT_CAPTURE_NOTIFICATION = ['ideal', 'blik', 'bankTransfer_IBAN', 'multibanco', 'dotpay', 'directEbanking', 'alipay', 'ebanking_FI'];
    const NOTIFY_AUTHORISATION_EMAIL_TEMPLATE = 'notify_authorisation';
    const XML_PATH_EMAIL_SENDER = 'trans_email/ident_sales/email';
    const XML_PATH_NAME_SENDER = 'trans_email/ident_sales/name';

    protected $connection;

    /**
     * @var ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * @var TransportBuilder
     */
    private $transportBuilder;

    public function __construct(
        Context              $context,
        Registry             $registry,
        ResourceConnection   $connection,
        ScopeConfigInterface $scopeConfig,
        TransportBuilder     $transportBuilder,
        AbstractResource     $resource = null,
        AbstractDb           $resourceCollection = null,
        array                $data = [])
    {
        parent::__construct(
            $context, $registry, $resource, $resourceCollection, $data);

        $this->connection = $connection->getConnection();
        $this->scopeConfig = $scopeConfig;
        $this->transportBuilder = $transportBuilder;
    }

    public function checkAuthorization()
    {
        if (!$this->scopeConfig->isSetFlag(self::CHECK_ADYEN_AUTHORIZATION_ENABLED_CONFIG_PATH)) {
            return;
        }

        $numberOfDaysToCheck = $this->scopeConfig->getValue(self::PAST_DAYS_TO_CHECK_TRANSACTIONS_CONFIG_PATH);
        $numberOfDaysToSendNotification = (int)$this->scopeConfig->getValue(self::NUMBER_DAYS_TO_SEND_NOTIFICATION_CONFIG_PATH);

        $adyenNotification = $this->connection->getTableName('adyen_notification');
        $salesOrder = $this->connection->getTableName('sales_order');
        $now = new DateTime();
        $dateFrom = new DateTime();
        $dateFrom = $dateFrom->sub(new DateInterval('P' . $numberOfDaysToCheck . 'D'));

        $queryAuth = "SELECT an.merchant_reference, an.pspreference, an.original_reference, an.created_at
                FROM {$adyenNotification} AS an
                INNER JOIN {$salesOrder} AS so
                ON so.increment_id = an.merchant_reference
                WHERE
                    so.status != 'manual_review' AND
                    an.event_code = 'AUTHORISATION'AND
                    an.success = 'true' AND
                    an.payment_method NOT IN ('" . implode("','", self::METHODS_WITHOUT_CAPTURE_NOTIFICATION) . "') AND
                    an.created_at >= '" . $dateFrom->format('Y-m-d') . "'";

        $authToCheck = $this->connection->fetchAll($queryAuth, [], Zend_Db::FETCH_OBJ);

        $queryCapture = "SELECT original_reference
                FROM {$adyenNotification}
                WHERE
                    event_code = 'CAPTURE'AND
                    success = 'true' AND
                    payment_method NOT IN ('" . implode("','", self::METHODS_WITHOUT_CAPTURE_NOTIFICATION) . "') AND
                    created_at >= '" . $dateFrom->format('Y-m-d') . "'";

        $captureToCheck = $this->connection->fetchAll($queryCapture, [], Zend_Db::FETCH_COLUMN);

        $queryCancel = "SELECT original_reference
                FROM {$adyenNotification}
                WHERE
                    event_code IN ('CANCELLATION', 'CANCEL_OR_REFUND') AND
                    success = 'true' AND
                    payment_method NOT IN ('" . implode("','", self::METHODS_WITHOUT_CAPTURE_NOTIFICATION) . "') AND
                    created_at >= '" . $dateFrom->format('Y-m-d') . "'";

        $cancelToCheck = $this->connection->fetchAll($queryCancel, [], Zend_Db::FETCH_COLUMN);

        foreach ($authToCheck as $key => $authNotify) {
            // Remove authorization if is present a cancellation with the same reference
            if (in_array($authNotify->pspreference, $cancelToCheck)) {
                unset($authToCheck[$key]);
            }
        }

        $orderToNotify = [];
        foreach ($authToCheck as $authNotify) {
            // Skip authorization if is present a capture with the same reference
            if (in_array($authNotify->pspreference, $captureToCheck)) {
                continue;
            }

            $authNotifyDate = new DateTime($authNotify->created_at);
            $interval = (int)$now->diff($authNotifyDate)->format('%d');
            if ($interval >= $numberOfDaysToSendNotification) {
                $orderToNotify[] = $authNotify->merchant_reference." (pspreference ".$authNotify->pspreference.")";
            }
        }

        if (empty($orderToNotify)) {
            return;
        }

        $transport = $this->transportBuilder->setTemplateIdentifier(
            self::NOTIFY_AUTHORISATION_EMAIL_TEMPLATE
        )
            ->setTemplateModel(BackendTemplate::class)
            ->setTemplateOptions([
                    'area' => Area::AREA_ADMINHTML,
                    'store' => Store::DEFAULT_STORE_ID
                ]
            )
            ->setTemplateVars([
                'incrementids' => implode(' - ', $orderToNotify),
                'days' => $numberOfDaysToSendNotification
            ])
            ->setFromByScope([
                'email' => $this->scopeConfig->getValue(self::XML_PATH_EMAIL_SENDER),
                'name' => $this->scopeConfig->getValue(self::XML_PATH_NAME_SENDER)
            ])
            ->addTo('payments@filoblu.com')
            ->getTransport();

        $transport->sendMessage();
    }
}
