<?php

namespace FiloBlu\OrderFlowRectifier\Model;

use FiloBlu\OrderFlowRectifier\Helper\AdyenBankTransferHelper;
use FiloBlu\OrderFlowRectifier\Model\Order\Email\Sender\OrderSender;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Data\Collection\AbstractDb;
use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Model\Context;
use Magento\Framework\Model\ResourceModel\AbstractResource;
use Magento\Framework\Registry;
use Magento\Framework\Serialize\SerializerInterface;
use Magento\Sales\Model\Order;
use Magento\Sales\Model\OrderRepository;

class AdyenBankTransferManagement extends AbstractModel
{
    protected $connection;

    /** @var AdyenBankTransferHelper */
    protected $bankTransferHelper;
    /** @var OrderRepository */
    protected $orderRepository;
    /** @var OrderSender */
    private $orderSender;
    /** @var SerializerInterface */
    private $serializer;

    /**
     * @param Context $context
     * @param Registry $registry
     * @param ResourceConnection $connection
     * @param AdyenBankTransferHelper $bankTransferHelper
     * @param OrderRepository $orderRepository
     * @param OrderSender $orderSender
     * @param SerializerInterface $serializer
     * @param AbstractResource|null $resource
     * @param AbstractDb|null $resourceCollection
     * @param array $data
     */
    public function __construct(
        Context                 $context,
        Registry                $registry,
        ResourceConnection      $connection,
        AdyenBankTransferHelper $bankTransferHelper,
        OrderRepository         $orderRepository,
        OrderSender             $orderSender,
        SerializerInterface     $serializer,
        AbstractResource        $resource = null,
        AbstractDb              $resourceCollection = null,
        array                   $data = [])
    {
        parent::__construct(
            $context, $registry, $resource, $resourceCollection, $data);

        $this->connection = $connection->getConnection();
        $this->bankTransferHelper = $bankTransferHelper;
        $this->orderRepository = $orderRepository;
        $this->orderSender = $orderSender;
        $this->serializer = $serializer;
    }

    public function ManageBankTransferOrders()
    {
        if (!$this->bankTransferHelper->automationBankTransferOrderIsEnabled()) {
            return true;
        }

        $limitDate = new \DateTime();
        $limitDate->setTimezone(new \DateTimeZone('Europe/Rome'));
        $limitDays = $this->bankTransferHelper->daysToDeleteOrder() + 1;
        $limitDate = $limitDate->sub(new \DateInterval('P' . $limitDays . 'D'));
        $salesOrder = $this->connection->getTableName('sales_order');
        $salesOrderPayment = $this->connection->getTableName('sales_order_payment');
        $orderStatus = $this->bankTransferHelper->getOrderStausToHandle();

        $query = "SELECT so.entity_id, so.increment_id
                FROM {$salesOrder} AS so
                INNER JOIN {$salesOrderPayment} AS sop
                ON so.entity_id = sop.parent_id
                WHERE
                    so.status IN ('".$orderStatus."') AND
                    so.customer_email != 'sistemi@filoblu.com' AND
                    (sop.method = 'adyen_bankTransfer_IBAN' OR
                    (sop.method = 'adyen_hpp'AND
                    sop.cc_type = 'bankTransfer_IBAN')) AND
                    sop.additional_information like '%" . AdyenBankTransferHelper::NOTIFY_BANKTRANSFER_DATA . "%' AND
                    so.created_at >= '" . $limitDate->format('Y-m-d') . "'";

        $queryOrder = $this->connection->fetchAll($query, [], \Zend_Db::FETCH_OBJ);

        if (empty($queryOrder)) {
            return true;
        }

        $adyenNotification = $this->connection->getTableName('adyen_notification');
        foreach ($queryOrder as $key => $orderData) {
            $entityId = $orderData->entity_id;
            $skipOrder = false;

            $queryNotification = "SELECT event_code FROM {$adyenNotification} WHERE merchant_reference = '".$orderData->increment_id."'";
            $adyenNotificationData = $this->connection->fetchAll($queryNotification, [], \Zend_Db::FETCH_ASSOC);

            // Added check to verify if there is authorization notification. If there is, the order has been paid and there is no need to send reminder emails
            if (count($adyenNotificationData) > 1) {
                foreach ($adyenNotificationData as $data) {
                    if ($data['event_code'] == 'AUTHORISATION') {
                        $skipOrder = true;
                    }
                }
            }

            if ($skipOrder) {
                continue;
            }

            $order = $this->orderRepository->get($entityId);
            $payment = $order->getPayment();
            $bankTransferAdditionalInfo = $this->serializer->unserialize($payment->getAdditionalInformation(AdyenBankTransferHelper::NOTIFY_BANKTRANSFER_DATA));

            $isFirstReminderToBeSent = $this->bankTransferHelper->checkFirstReminder($bankTransferAdditionalInfo);
            $isSecondReminderToBeSent = $this->bankTransferHelper->checkSecondReminder($bankTransferAdditionalInfo);
            $isOrderToBeDeleted = $this->bankTransferHelper->checkOrderDelete($bankTransferAdditionalInfo);

            if ($isFirstReminderToBeSent) {
                // Flag used to select the correct email template
                $payment->setAdditionalInformation(AdyenBankTransferHelper::FIRST_REMINDER_INDEX, '1');
                $sendReminder = $this->orderSender->sendReminder($order);
                if ($sendReminder) {
                    $bankTransferAdditionalInfo[AdyenBankTransferHelper::FIRST_REMINDER_INDEX][AdyenBankTransferHelper::SENT_INDEX] = 1;
                    $payment->setAdditionalInformation(AdyenBankTransferHelper::NOTIFY_BANKTRANSFER_DATA, $this->serializer->serialize($bankTransferAdditionalInfo));
                    $order->addCommentToStatusHistory('Bank Transfer Tool: first reminder notification sent.');
                } else {
                    $order->addCommentToStatusHistory('Bank Transfer Tool: error sending first reminder notification.');
                }
                // Remove flag used for select email template
                $payment->unsAdditionalInformation(AdyenBankTransferHelper::FIRST_REMINDER_INDEX);
                $this->orderRepository->save($order);
            } elseif ($isSecondReminderToBeSent) {
                // Flag used to select the correct email template
                $payment->setAdditionalInformation(AdyenBankTransferHelper::SECOND_REMINDER_INDEX, '1');
                $sendReminder = $this->orderSender->sendReminder($order);
                if ($sendReminder) {
                    $bankTransferAdditionalInfo[AdyenBankTransferHelper::SECOND_REMINDER_INDEX][AdyenBankTransferHelper::SENT_INDEX] = 1;
                    $payment->setAdditionalInformation(AdyenBankTransferHelper::NOTIFY_BANKTRANSFER_DATA, $this->serializer->serialize($bankTransferAdditionalInfo));
                    $order->addCommentToStatusHistory('Bank Transfer Tool: second reminder notification sent.');
                } else {
                    $order->addCommentToStatusHistory('Bank Transfer Tool: error sending second reminder notification.');
                }
                // Remove flag used for select email template
                $payment->unsAdditionalInformation(AdyenBankTransferHelper::SECOND_REMINDER_INDEX);
                $this->orderRepository->save($order);
            } elseif ($isOrderToBeDeleted) {
                if (!$order->canCancel()) {
                    $order->setState(Order::STATE_NEW);
                }
                $order->cancel();
                $order->addCommentToStatusHistory('Bank Transfer Tool: order deleted.');
                $this->orderRepository->save($order);
            }
        }

        return true;
    }
}
