<?php

namespace FiloBlu\OrderFlowRectifier\Helper;

use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Quote\Model\QuoteRepository;
use Magento\Quote\Model\Quote\Item\Repository;
use Magento\Store\Model\ScopeInterface;

class Data
{
    const CHECK_QUOTE_ITEM_ENABLED_CONFIG_PATH = 'order_flow_rectifier/quote_item/quote_item_enable';
    const EMAIL_LIST_CONFIG_PATH = 'order_flow_rectifier/quote_item/email_notification';
    const DELETE_QUOTE_ITEM_CONFIG_PATH = 'order_flow_rectifier/quote_item/delete_quote_item';
    const XML_PATH_STORE_NAME = 'general/store_information/name';
    const PRODUCT_TYPE_CONFIGURABLE = 'configurable';
    const SCHEDULED_FIX_CCTYPE_ENABLED_CONFIG_PATH = 'order_flow_rectifier/general/scheduled_cctype_check_enabled';
    const SCHEDULED_CANCEL_TEST_ORDER_CC_ENABLED_CONFIG_PATH = 'order_flow_rectifier/general/scheduled_cancel_test_order_cc_enabled';
    const QUOTE_CCTYPE_ENABLED_CONFIG_PATH = 'order_flow_rectifier/quote_cc_type/quote_cc_type_enable';
    const QUOTE_CCTYPE_PAYMENT_METHOD_TO_FIX_CONFIG_PATH = 'order_flow_rectifier/quote_cc_type/payment_method';
    const QUOTE_CCTYPE_OTHER_PAYMENT_METHOD_TO_FIX_CONFIG_PATH = 'order_flow_rectifier/quote_cc_type/other_payment_method';
    const PAYMENT_METHOD_ADYEN_CC = 'adyen_cc';
    const PAYMENT_METHOD_ADYEN_HPP = 'adyen_hpp';
    const PAYMENT_METHOD_ADYEN_PBL = 'adyen_pay_by_link';
    const KLARNA_PAY_LATER_CODE = 'klarna_paylater';
    const KLARNA_CODE = 'klarna';
    const KLARNA_ACCOUNT_CODE = 'klarna_account';
    const CC_TYPE_MAP = array(
        "amex" => "AE",
        "visa" => "VI",
        "mc" => "MC",
        "discover" => "DI",
        "jcb" => "JCB",
        "diners" => "DN",
        "maestro" => "MI",
        "cup" => "UN",
        "bcmc" => "BCMC",
        "hipercard" => "HIPERCARD",
        "elo" => "ELO",
        "troy" => "TROY",
        "dankort" => "DANKORT",
        "cartebancaire" => "CB",
        "korean_local_card" => "KCP"
    );
    const CC_TYPE_HPP_MAP = array(
        'maestro_applepay' => 'MAESTROAPPLEPAY',
        'amex_applepay' => 'AMEXAPPLEPAY',
        'mc_applepay' => 'MCAPPLEPAY',
        'visa_applepay' => 'VIAPPLEPAY',
        'discover_applepay' => 'DISCOVERAPPLEPAY',
        'jcb_applepay' => 'JCBAPPLEPAY',
        'electron_applepay' => 'ELECTRONAPPLEPAY',
        'elo_applepay' => 'ELOAPPLEPAY',
        'elodebit_applepay' => 'ELODEBITAPPLEPAY',
        'interac_applepay' => 'INTERACAPPLEPAY',
        'girocard_applepay' => 'GIROCARDAPPLEPAY',
        'mc_clicktopay' => 'CLICKTOPAYMC',
        'visa_clicktopay' => 'CLICKTOPAYVISA',
        'mcstandarddebit' => 'MCSTANDARDDEBIT'
    );
    const COUNTRY_KLARNA_PAY_OVER_TIME_MAP = array(
        'IT' => 'klarna_payinthree',
        'ES' => 'klarna_payinthree',
        'FR' => 'klarna_payinthree',
        'NL' => 'klarna_payinthree',
        'GB' => 'klarna_payinthree',
        'PT' => 'klarna_payinthree',
        'IE' => 'klarna_payinthree',
        'DE' => 'klarna_financing',
        'AT' => 'klarna_financing',
        'FI' => 'klarna_financing'
    );
    const NOTIFY_BANK_TRANSFER_AUTHORIZATION_ENABLE_PATH = 'order_flow_rectifier/check_adyen_authorization/notify_bank_transfer_authorization_enable';
    const NOTIFY_BANK_TRANSFER_AUTHORIZATION_RECIPIENTS_PATH = 'order_flow_rectifier/check_adyen_authorization/notify_bank_transfer_authorization_recipients';
    const XML_PATH_EMAIL_SENDER = 'trans_email/ident_sales/email';
    const XML_PATH_NAME_SENDER = 'trans_email/ident_sales/name';

    protected $config;
    protected $quoteRepository;
    protected $emailNotifyHelper;
    protected $quoteItemRepository;

    public function __construct(
        ScopeConfigInterface $config,
        QuoteRepository $quoteRepository,
        EmailNotification $emailNotifyHelper,
        Repository $quoteItemRepository
    ) {
        $this->config = $config;
        $this->quoteRepository = $quoteRepository;
        $this->emailNotifyHelper = $emailNotifyHelper;
        $this->quoteItemRepository = $quoteItemRepository;
    }

    public function checkQuoteItem($quoteItem, $quoteId)
    {
        $missingQuoteChildren = array();
        $itemData = array();

        foreach ($quoteItem as $key => $item)
        {
            if($item->getProductType() == self::PRODUCT_TYPE_CONFIGURABLE)
            {
                $itemHasChildren = false;
                $itemId = $item->getItemId();
                $quoteItemChildren = $item->getChildren();
                if($quoteItemChildren && !empty($quoteItemChildren))
                {
                    foreach ($quoteItemChildren as $children)
                    {
                        if ($children->getParentItemId() == $itemId)
                        {
                            $itemHasChildren = true;
                        }
                    }
                }

                if (!$itemHasChildren)
                {
                    $productName = (!empty($item->getName()) && !is_null($item->getName())) ? $item->getName() : '';
                    $itemData[] = array(
                        'item_id' => $itemId,
                        'sku' => $item->getSku(),
                        'product_id' => $item->getProductId(),
                        'product_name' => $productName
                    );
                }
            }
        }

        if(!empty($itemData))
        {
            $missingQuoteChildren['quote_id'] = $quoteId;
            $missingQuoteChildren['item_data'] = $itemData;
        }

        return $missingQuoteChildren;
    }

    public function getQuoteItem($quoteId)
    {
        $quote = $this->quoteRepository->get($quoteId);
        return $quote->getItems();
    }

    public function checkQuoteItemIsEnabled()
    {
        return $this->config->isSetFlag(self::CHECK_QUOTE_ITEM_ENABLED_CONFIG_PATH);
    }

    public function deleteQuoteItemIsEnabled()
    {
        return $this->config->isSetFlag(self::DELETE_QUOTE_ITEM_CONFIG_PATH);
    }

    public function deleteQuoteItem($quoteId, $itemId)
    {
        $this->quoteItemRepository->deleteById($quoteId, $itemId);
    }

    public function sendNotificationMail($missingData, $action = '')
    {
        $emailAddressString = $this->config->getValue(self::EMAIL_LIST_CONFIG_PATH);

        if (!is_null($emailAddressString) && $emailAddressString && $emailAddressString !== '') {
            $emailAddress = explode(',', $emailAddressString);
            $emailAddress = array_map('trim', $emailAddress);

            $storeName = $this->config->getValue(self::XML_PATH_STORE_NAME);

            $notificationString = $action != '' ? 'Missing items found during: '.$action : '';
            $notificationString.= '<ul>';
            foreach ($missingData['item_data'] as $val)
            {
                $notificationString.= '<li>missing simple for item_id -> '.$val['item_id'].' (SKU '.$val['sku'].' - product id '.$val['product_id'].' - name '.$val['product_name'].')</li>';
            }
            $notificationString.= '</ul>';

            // Datas for email
            $data = [
                'mail_subject' => 'Quote item missing for ' . $storeName,
                'quote' => $missingData['quote_id'],
                'errors' => $notificationString
            ];

            $this->emailNotifyHelper->notify($emailAddress, $data);
        }
    }

    public function bankTransferAuthNotificationIsEnabled()
    {
        return $this->config->isSetFlag(self::NOTIFY_BANK_TRANSFER_AUTHORIZATION_ENABLE_PATH);
    }

    public function getBankTransferAuthNotificationRecipients()
    {
        return $this->config->getValue(self::NOTIFY_BANK_TRANSFER_AUTHORIZATION_RECIPIENTS_PATH, ScopeInterface::SCOPE_STORE);
    }

    /**
     * @return array|false|string[]
     */
    public function getBankTransferAuthNotificationRecipientsArray()
    {
        $result = [];
        $recipients = $this->getBankTransferAuthNotificationRecipients();
        if($recipients != '')
        {
            $result = explode(',', $recipients);
        }
        return $result;
    }

    public function emailSender()
    {
        return $this->config->getValue(self::XML_PATH_EMAIL_SENDER);
    }

    public function nameSender()
    {
        return $this->config->getValue(self::XML_PATH_NAME_SENDER);
    }
}
