<?php


namespace FiloBlu\OrderFlowRectifier\Console\Command;

use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

/**
 * Class Process
 *
 * @package FiloBlu\OrderFlowRectifier\Console\Command
 */
class Process extends Command
{
    const DRY_RUN_OPTION = "dry-run";
    const SINGLE_ORDER_OPTION = "order";
    const SINGLE_RECTIFIER_TO_RUN_ARGUMENT = "rectifier";

    protected $rectifyProcessorFactory;
    protected $rectifierPoolFactory;

    public function __construct(
        \FiloBlu\OrderFlowRectifier\Model\ProcessorFactory $rectifyProcessorFactory,
        \FiloBlu\OrderFlowRectifier\Model\RectifierPoolFactory $rectifierPoolFactory
    )
    {
        $this->rectifyProcessorFactory = $rectifyProcessorFactory;
        $this->rectifierPoolFactory = $rectifierPoolFactory;
        parent::__construct();
    }

    /**
     * {@inheritdoc}
     */
    protected function execute(
        InputInterface $input,
        OutputInterface $output
    ) {
        $isDryRunMode = $input->getOption(self::DRY_RUN_OPTION);
        $singleRectifierToRun = $input->getArgument(self::SINGLE_RECTIFIER_TO_RUN_ARGUMENT);
        //rectifiers list comes from di.xml
        if (!is_null($singleRectifierToRun)) {
            $rectifiers = [$singleRectifierToRun => $this->rectifierPoolFactory->create()->get($singleRectifierToRun) ];
        } else {
            $rectifiers = $this->rectifierPoolFactory->create()->getList();
        }
        $rectifiersCount = count($rectifiers);
        $output->writeln('<info>Found ' . $rectifiersCount . ' rectifier(s)</info>');
        $counter = 1;
        $singleOrder = $input->getOption(self::SINGLE_ORDER_OPTION);
        foreach ($rectifiers as $code => $rectifier) {
            $output->writeln('<info>Rectifier ' . $counter .'/' . $rectifiersCount . ': ' . $code .' (' . $rectifier->getTitle() . ')</info>');
            $ordersToProcess = $rectifier->loadOrders();
            $output->writeln('Found ' . count($ordersToProcess) . ' orders to process');
            foreach ($ordersToProcess as $order) {
                if (!is_null($singleOrder) && $order->increment_id != $singleOrder) {
                    continue;
                }
                $output->write('Processing order ' . $order->increment_id . ' ... ');
                if ($isDryRunMode) {
                    $output->writeln('Skipped due to dry run');
                } else {
                    $success = $rectifier->executeSingle($order);
                    if ($success) {
                        $output->writeln('Success');
                    } else {
                        $output->writeln('Not processed');
                    }
                }
            }
            $counter++;
        }
        $output->writeln('DONE');
    }

    /**
     * {@inheritdoc}
     */
    protected function configure()
    {
        $this->setName("filoblu:orderflowrectifier:process");
        $this->setDescription("Find orders with anomalies and rectify them");
        $this->setDefinition([
            new InputArgument(self::SINGLE_RECTIFIER_TO_RUN_ARGUMENT, InputArgument::OPTIONAL, "Single rectifier to run"),
            new InputOption(self::DRY_RUN_OPTION, "-d", InputOption::VALUE_NONE, "Do not save modifications in database"),
            new InputOption(self::SINGLE_ORDER_OPTION, "-o", InputOption::VALUE_REQUIRED, "Rectify a single order by specifying the increment_id"),
        ]);
        parent::configure();
    }
}
