<?php

namespace FiloBlu\Newsletter\Model\Import;

use Exception;
use FiloBlu\Newsletter\Helper\Import;
use InvalidArgumentException;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\ValidatorException;
use Magento\Framework\Json\Helper\Data;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\ImportExport\Model\Import as ImportExport;
use Magento\ImportExport\Model\Import\Entity\AbstractEntity;
use Magento\ImportExport\Model\Import\ErrorProcessing\ProcessingErrorAggregatorInterface;
use Magento\ImportExport\Model\ResourceModel\Helper;

/**
 * Class Subscriber
 * @package FiloBlu\Newsletter\Model\Import
 */
class Subscriber extends AbstractEntity
{

    /**
     * Helper
     *
     * @var Import
     */
    protected $_helper;
    protected $logInHistory = true;
    protected $needColumnCheck;

    protected $_permanentAttributes = [
        'store_code',
        'email',
        'status'
    ];

    protected $validColumnNames = [
        'group_name',
        'store_code',
        'email',
        'status'
    ];

    /**
     * @var ResourceConnection
     */
    protected $_resource;

    /**
     * @var
     */
    private $serializer;

    /**
     * Subscriber constructor.
     * @param Data $jsonHelper
     * @param \Magento\ImportExport\Helper\Data $importExportData
     * @param \Magento\ImportExport\Model\ResourceModel\Import\Data $importData
     * @param ResourceConnection $resource
     * @param Helper $resourceHelper
     * @param ProcessingErrorAggregatorInterface $errorAggregator
     * @param Import $helper
     */
    public function __construct(
        Data $jsonHelper,
        \Magento\ImportExport\Helper\Data $importExportData,
        \Magento\ImportExport\Model\ResourceModel\Import\Data $importData,
        ResourceConnection $resource,
        Helper $resourceHelper,
        ProcessingErrorAggregatorInterface $errorAggregator,
        Import $helper
    )
    {
        $this->jsonHelper = $jsonHelper;
        $this->_importExportData = $importExportData;
        $this->_resourceHelper = $resourceHelper;
        $this->_dataSourceModel = $importData;
        $this->_resource = $resource;
        $this->_connection = $resource->getConnection(ResourceConnection::DEFAULT_CONNECTION);
        $this->errorAggregator = $errorAggregator;
        $this->_helper = $helper;
        $this->getAllowedColumns();
    }

    protected function getAllowedColumns()
    {

        $fieldsMapping = unserialize($this->_helper->getConfig('general/fields_mapping'));

        if ($fieldsMapping && count($fieldsMapping)) {

            foreach ($fieldsMapping as $field) {
                $this->validColumnNames[] = $field['dbcolumnname'];
            }
        }
    }

    /**
     * Validate data.
     *
     * @return ProcessingErrorAggregatorInterface
     * @throws LocalizedException
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     */
    public function validateData()
    {
        if (!$this->_dataValidated) {
            $this->getErrorAggregator()->clear();
            // do all permanent columns exist?
            $absentColumns = array_diff($this->_permanentAttributes, $this->getSource()->getColNames());
            $this->addErrors(self::ERROR_CODE_COLUMN_NOT_FOUND, $absentColumns);

            if (ImportExport::BEHAVIOR_DELETE != $this->getBehavior()) {
                // check attribute columns names validity
                $columnNumber = 0;
                $emptyHeaderColumns = [];
                $invalidColumns = [];
                $invalidAttributes = [];
                foreach ($this->getSource()->getColNames() as $columnName) {
                    $columnNumber++;
                    if (!$this->isAttributeParticular($columnName)) {
                        if (trim($columnName) == '') {
                            $emptyHeaderColumns[] = $columnNumber;
                        } elseif (!preg_match('/^[a-z][a-z0-9_]*$/', $columnName)) {
                            $invalidColumns[] = $columnName;
                        } elseif ($this->needColumnCheck && !in_array($columnName, $this->getValidColumnNames())) {
                            $invalidAttributes[] = $columnName;
                        }
                    }
                }
                $this->addErrors(self::ERROR_CODE_INVALID_ATTRIBUTE, $invalidAttributes);
                $this->addErrors(self::ERROR_CODE_COLUMN_EMPTY_HEADER, $emptyHeaderColumns);
                $this->addErrors(self::ERROR_CODE_COLUMN_NAME_INVALID, $invalidColumns);
            }

            if (!$this->getErrorAggregator()->getErrorsCount()) {
                $this->_saveValidatedBunches();
                $this->_dataValidated = true;
            }
        }
        return $this->getErrorAggregator();
    }

    /**
     * Validate data rows and save bunches to DB.
     *
     * @return $this
     * @throws LocalizedException
     * @throws ValidatorException
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     */
    protected function _saveValidatedBunches()
    {
        $source = $this->_getSource();
        $currentDataSize = 0;
        $bunchRows = [];
        $startNewBunch = false;
        $nextRowBackup = [];
        $maxDataSize = $this->_resourceHelper->getMaxDataSize();
        $bunchSize = $this->_importExportData->getBunchSize();
        $skuSet = [];

        $source->rewind();
        $this->_dataSourceModel->cleanBunches();

        while ($source->valid() || $bunchRows) {
            if ($startNewBunch || !$source->valid()) {
                $this->_dataSourceModel->saveBunch($this->getEntityTypeCode(), $this->getBehavior(), $bunchRows);

                $bunchRows = $nextRowBackup;
                $currentDataSize = strlen($this->getSerializer()->serialize($bunchRows));
                $startNewBunch = false;
                $nextRowBackup = [];
            }
            if ($source->valid()) {
                try {
                    $rowData = $source->current();
                    $skuSet[$rowData['email']] = true;
                } catch (InvalidArgumentException $e) {
                    $this->addRowError($e->getMessage(), $this->_processedRowsCount);
                    $this->_processedRowsCount++;
                    $source->next();
                    continue;
                }

                $this->_processedRowsCount++;

                if ($this->validateRow($rowData, $source->key())) {
                    // add row to bunch for save
                    $rowData = $this->_prepareRowForDb($rowData);
                    $rowSize = strlen($this->jsonHelper->jsonEncode($rowData));

                    $isBunchSizeExceeded = $bunchSize > 0 && count($bunchRows) >= $bunchSize;

                    if ($currentDataSize + $rowSize >= $maxDataSize || $isBunchSizeExceeded) {
                        $startNewBunch = true;
                        $nextRowBackup = [$source->key() => $rowData];
                    } else {
                        $bunchRows[$source->key()] = $rowData;
                        $currentDataSize += $rowSize;
                    }
                }
                $source->next();
            }
        }
        $this->_processedEntitiesCount = count($skuSet);

        return $this;
    }

    /**
     * Entity type code getter.
     *
     * @return string
     */
    public function getEntityTypeCode()
    {
        return 'newsletter_subscriber_import';
    }

    /**
     * Get Serializer instance
     *
     * Workaround. Only way to implement dependency and not to break inherited child classes
     *
     * @return Json
     * @deprecated 100.2.0
     */
    private function getSerializer()
    {
        if (null === $this->serializer) {
            $this->serializer = ObjectManager::getInstance()->get(Json::class);
        }
        return $this->serializer;
    }

    public function validateRow(array $rowData, $rowNum)
    {
        return true;
    }

    /**
     * Create Advanced price data from raw data.
     *
     * @return bool Result of operation.
     * @throws Exception
     */
    protected function _importData()
    {
//
//          \Magento\Framework\App\ObjectManager::getInstance()
//                ->get('Psr\Log\LoggerInterface')->debug(print_r($this->getBehavior(), true));

//        if (\Magento\ImportExport\Model\Import::BEHAVIOR_CUSTOM == $this->getBehavior()) {
        $this->updateData();
//        }

        return true;
    }

    protected function updateData()
    {
        $behavior = $this->getBehavior();
        $i = 0;

        while ($bunch = $this->_dataSourceModel->getNextBunch()) {

            $entityList = [];
            foreach ($bunch as $rowNum => $rowData) {


                if (!$this->validateRow($rowData, $rowNum)) {
//                    $this->addRowError(ValidatorInterface::ERROR_TITLE_IS_EMPTY, $rowNum);
                    continue;
                }
                if ($this->getErrorAggregator()->hasToBeTerminated()) {
                    $this->getErrorAggregator()->addRowToSkip($rowNum);
                    continue;
                }

                $i++;

                $entityList[$i]['group_name'] = $rowData['group_name'];
                $entityList[$i]['store_code'] = $rowData['store_code'];
                $entityList[$i]['email'] = $rowData['email'];
                $entityList[$i]['status'] = $rowData['status'];


                foreach ($this->validColumnNames as $column) {
                    if (isset($rowData[$column])) {
                        $entityList[$i][$column] = $rowData[$column];
                    }
                }
            }

            $this->countItemsCreated += $i;

//            if (\Magento\ImportExport\Model\Import::BEHAVIOR_APPEND == $behavior) {
//                $this->saveEntityFinish($entityList, self::TABLE_Entity);
//            }
            $this->_helper->importDbData($entityList);
        }

//            \Magento\Framework\App\ObjectManager::getInstance()
//                ->get('Psr\Log\LoggerInterface')->debug(print_r($entityList, true));

        return $this;
    }

}
