<?php

/**
 *
 * Copyright © 2016 Magento. All rights reserved.
 * See COPYING.txt for license details.
 */

namespace FiloBlu\Newsletter\Controller\Subscriber;

use Exception;
use FiloBlu\Newsletter\Helper\Data;
use FiloBlu\Newsletter\Model\PreConfirm;
use FiloBlu\Newsletter\Model\ThankYou;
use Magento\Framework\Controller\Result\Redirect;
use Magento\Framework\Controller\ResultFactory;
use Magento\Framework\Controller\ResultInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\View\Result\Layout;
use Magento\Newsletter\Model\Subscriber;

/**
 * Class NewLandingAction
 * @package FiloBlu\Newsletter\Controller\Subscriber
 */
class NewLandingAction extends \Magento\Newsletter\Controller\Subscriber\NewAction
{

    /**
     * @return Redirect|ResultInterface|Layout|null
     */
    public function execute()
    {
        /** @var Data $helper */
        $helper = $this->_objectManager->get(Data::class);
        /** @var PreConfirm $modelPreConfirm */
        $modelPreConfirm = $this->_objectManager->get(PreConfirm::class);
        /** @var ThankYou $modelThankYou */
        $modelThankYou = $this->_objectManager->get(ThankYou::class);

        if ($helper->getConfig('general/enabled') && $this->getRequest()->isPost() && $this->getRequest()->getPost('email')) {
            $resultJson = $this->resultFactory->create(ResultFactory::TYPE_JSON);

            $email = (string)$this->getRequest()->getPost('email');

            try {
                $this->validateEmailFormat($email);
                $this->validateGuestSubscription();
                $this->validateEmailAvailable($email);
                $baseUrl = $this->_storeManager->getStore()->getBaseUrl();
                $status = $this->_subscriberFactory->create()->subscribe($email);
                $preConfirmId = $this->getRequest()->getPost('preconfirm_id');
                $thankYouId = $this->getRequest()->getPost('thankyou_id');

                if ($status == Subscriber::STATUS_NOT_ACTIVE) {
                    if ($preConfirmId && $preConfirmId != 0) {
                        $helper->setPreConfirmCookie($preConfirmId);
                        $resultJson->setData([
                            'success'  => true,
                            'error'    => false,
                            'redirect' => $baseUrl . $modelPreConfirm
                                    ->getUrlPreConfirmPageById($preConfirmId)
                                    ->getData()['url_key']]);
                    } else {
                        $resultJson->setData([
                            'success'  => true,
                            'error'    => false,
                            'message'  => __('The confirmation request has been sent.'),
                            'redirect' => $baseUrl]);
                    }
                } else {
                    $subscriber = $this->_subscriberFactory->create()->loadByEmail($email);
                    if ($thankYouId && $thankYouId != 0) {
                        $helper->setThankYouCookie($thankYouId);
                        $resultJson->setData([
                            'subscriber_id' => $subscriber->getId(),
                            'success'       => true,
                            'error'         => false,
                            'redirect'      => $baseUrl . $modelThankYou
                                    ->getUrlThankYouPageById($thankYouId)
                                    ->getData()['url_key']]);
                    } else {
                        $resultJson->setData([
                            'subscriber_id' => $subscriber->getId(),
                            'success'       => true,
                            'error'         => false,
                            'message'       => __('Thank you for your subscription.'),
                            'redirect'      => $baseUrl]);
                    }
                }
            } catch (LocalizedException $e) {
                $resultJson->setData([
                    'success' => false,
                    'error'   => true,
                    'message' => __('There was a problem with the subscription: %1', $e->getMessage())]);
            } catch (Exception $e) {
                $resultJson->setData([
                    'success' => false,
                    'error'   => true,
                    'message' => __($e->getMessage())]);
            }

            return $resultJson;
        }

        return null;
    }

    /**
     * Validates that the email address isn't being used by a different account.
     *
     * @param string $email
     * @return void
     * @throws LocalizedException
     */
    protected function validateEmailAvailable($email)
    {
        $websiteId = $this->_storeManager->getStore()->getWebsiteId();
        if ($this->_customerSession->getCustomerDataObject()->getEmail() !== $email && !$this->customerAccountManagement->isEmailAvailable($email, $websiteId)
        ) {
            throw new LocalizedException(
                __('This email address is already assigned to another user.')
            );
        }

        $subscriber = $this->_subscriberFactory->create()->loadByEmail($email);

        if ($subscriber->getSubscriberStatus() == Subscriber::STATUS_SUBSCRIBED) {
            throw new LocalizedException(
                __('This email address is already assigned to another user.')
            );
        }
    }
}
