<?php

namespace FiloBlu\Newsletter\Controller\Subscriber;

use Exception;
use FiloBlu\Newsletter\Helper\Data;
use FiloBlu\Newsletter\Model\LandingPageFactory;
use FiloBlu\Newsletter\Model\PreConfirm;
use FiloBlu\Newsletter\Model\ThankYou;
use Magento\Customer\Api\AccountManagementInterface;
use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Customer\Controller\Account\CreatePost;
use Magento\Customer\Model\Account\Redirect as AccountRedirect;
use Magento\Customer\Model\CustomerRegistry;
use Magento\Customer\Model\Data\Customer;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\Controller\Result\Json;
use Magento\Framework\Controller\Result\JsonFactory;
use Magento\Framework\Data\Form\FormKey\Validator;
use Magento\Framework\Exception\InputException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\StateException;
use Magento\Framework\Stdlib\Cookie\CookieMetadataFactory;
use Magento\Framework\Stdlib\Cookie\PhpCookieManager;
use Magento\Store\Model\StoreManagerInterface;
use RuntimeException;

/**
 * Class NewLandingRegisterAction
 * @package FiloBlu\Newsletter\Controller\Subscriber
 */
class NewLandingRegisterAction extends CreatePost
{
    /**
     * @var \FiloBlu\Newsletter\Model\LandingPage|null
     */
    protected $landingPage;

    /**
     * @return Json|void|null
     */
    public function execute()
    {
        $objectManager = ObjectManager::getInstance();
        $resultJsonFactory = $objectManager->get(JsonFactory::class);
        $helper = $objectManager->create(Data::class);
        $accountRedirect = $objectManager->create(AccountRedirect::class);
        $formKeyValidator = $objectManager->get(Validator::class);
        $cookieMetadataFactory = $objectManager->get(CookieMetadataFactory::class);
        $cookieMetadataManager = $objectManager->get(PhpCookieManager::class);
        /** @var ScopeConfigInterface $scopeConfig */
        $scopeConfig = $objectManager->get(ScopeConfigInterface::class);
        /** @var StoreManagerInterface $storeManager */
        $storeManager = $objectManager->get(StoreManagerInterface::class);
        /** @var ThankYou $thankYou */
        $thankYou = $objectManager->get(ThankYou::class);
        /** @var PreConfirm $preConfirm */
        $preConfirm = $objectManager->get(PreConfirm::class);
        /** @var CustomerRegistry $customerRegistry */
        $customerRegistry = $objectManager->get(CustomerRegistry::class);
        /** @var CustomerRepositoryInterface $customerRepository */
        $customerRepository = $objectManager->get(CustomerRepositoryInterface::class);

        $newCustomer = false;
        $allowExistingCustomer = $this->allowExistingCustomers();

        if (!$helper->getConfig('general/enabled') || !$this->getRequest()->isPost()) {
            return null;
        }

        $resultJson = $resultJsonFactory->create();
        $this->session->regenerateId();
        $baseUrl = $storeManager->getStore()->getBaseUrl();

        if (!$formKeyValidator->validate($this->getRequest())) {
            return $resultJson->setData(['success' => false, 'error' => true, 'message' => __('Invalid form key.')]);
        }

        if (!$this->registration->isAllowed()) {
            return $resultJson->setData(['success' => false, 'error' => true, 'message' => __('Registration is not allowed.')]);
        }

        try {

            $address = $this->extractAddress();
            $addresses = $address === null ? [] : [$address];
            $customer = $this->customerExtractor->extract('customer_account_create', $this->_request);

            if ($this->accountManagement->isEmailAvailable($customer->getEmail())) {
                // Register Customer
                $newCustomer = true;

                $customer->setAddresses($addresses);
                $password = $this->getRequest()->getParam('password');
                $confirmation = $this->getRequest()->getParam('password_confirmation');
                $redirectUrl = $this->session->getBeforeAuthUrl();
                $this->checkPasswordConfirmation($password, $confirmation);
                $customer = $this->accountManagement->createAccount($customer, $password, $redirectUrl);

                if ($this->getRequest()->getParam('is_subscribed', false)) {
                    $this->subscriberFactory->create()->subscribeCustomerById($customer->getId());
                }

                $this->_eventManager->dispatch('customer_register_success', ['account_controller' => $this, 'customer' => $customer]);
            }

            /** @var Customer $customer */
            $customer = $customerRegistry->retrieveByEmail($customer->getEmail(), $storeManager->getStore()->getWebsiteId());
            $landingPage = $this->getLandingPage();

            if ((bool)$landingPage->getAssignCustomerGroup() && $landingPage->getCustomerGroup()) {
                $customer->setGroupId((int)$landingPage->getCustomerGroup())->save();
            }

            $confirmationStatus = $this->accountManagement->getConfirmationStatus($customer->getId());

            if ($confirmationStatus === AccountManagementInterface::ACCOUNT_CONFIRMATION_REQUIRED) {
                if ($this->getRequest()->getPost('preconfirm_id') && (int)$this->getRequest()->getPost('preconfirm_id') !== 0) {
                    $helper->setPreConfirmCookie($this->getRequest()->getPost('preconfirm_id'));
                    return $resultJson->setData([
                        'success'  => true,
                        'error'    => false,
                        'redirect' => $baseUrl . $preConfirm->getUrlPreConfirmPageById($this->getRequest()->getPost('preconfirm_id'))->getData()['url_key']
                    ]);
                }

                $email = $this->customerUrl->getEmailConfirmationUrl($customer->getEmail());
                $this->messageManager->addSuccessMessage(
                    __('You must confirm your account. Please check your email for the confirmation link or <a href="%1">click here</a> for a new link.', $email)
                );
                return $resultJson->setData(['success' => true, 'error' => false, 'redirect' => $baseUrl]);

            } else {

                if ($newCustomer) {
                    $this->session->setCustomerDataAsLoggedIn($customerRepository->getById($customer->getId()));
                } else {

                    if($allowExistingCustomer) {
                        try {
                            $this->accountManagement->authenticate($customer->getEmail(), $this->getRequest()->getParam('password'));
                            $this->session->setCustomerDataAsLoggedIn($customerRepository->getById($customer->getId()));

                        } catch (\Throwable $throwable) {
                            return $resultJson->setData(['success' => false, 'error' => true, 'message' => __('Invalid password')]);
                        }
                    } else {
                        return $resultJson->setData([
                            'success' => false,
                            'error' => true,
                            'message' =>  __('There is already an account with this email address. If you are sure that it is your email address, <a href="%1">click here</a> to get your password and access your account.',
                            $this->urlModel->getUrl('customer/account/forgotpassword')
                        )]);
                    }
                }

                $thankyouId = $this->getRequest()->getPost('thankyou_id');

                if ($thankyouId && (int)$thankyouId !== 0) {
                    $helper->setThankYouCookie($thankyouId);
                    return $resultJson->setData([
                        'success'  => true,
                        'error'    => false,
                        'redirect' => $baseUrl . $thankYou->getUrlThankYouPageById($thankyouId)->getData()['url_key']]);
                } else {
                    $this->messageManager->addSuccessMessage($this->getSuccessMessage());
                    $requestedRedirect = $accountRedirect->getRedirectCookie();
                    if ($requestedRedirect && !$scopeConfig->getValue('customer/startup/redirect_dashboard')) {
                        $accountRedirect->clearRedirectCookie();
                        $resultJson->setData(['success' => true, 'error' => false, 'redirect' => $this->_redirect->success($requestedRedirect)]);
                    } else {
                        $resultJson->setData(['success' => true, 'error' => false, 'redirect' => $baseUrl]);
                    }
                }
            }

            if ($cookieMetadataManager->getCookie('mage-cache-sessid')) {
                $metadata = $cookieMetadataFactory->createCookieMetadata();
                $metadata->setPath('/');
                $cookieMetadataManager->deleteCookie('mage-cache-sessid', $metadata);
            }

        } catch (StateException $e) {
            // TODO : This condition is never reached
            $url = $this->urlModel->getUrl('customer/account/forgotpassword');
            // @codingStandardsIgnoreStart
            $message = __(
                'There is already an account with this email address. If you are sure that it is your email address, <a href="%1">click here</a> to get your password and access your account.',
                $url
            );
            // @codingStandardsIgnoreEnd

            $resultJson->setData(['success' => false, 'error' => true, 'message' => $message]);
        } catch (InputException $e) {
            $messages = __($e->getMessage());
            foreach ($e->getErrors() as $error) {
                $messages .= '<br>' . __($error->getMessage());
            }
            $resultJson->setData(['success' => false, 'error' => true, 'message' => $messages]);
        } catch (LocalizedException $e) {
            $resultJson->setData(['success' => false, 'error' => true, 'message' => __($e->getMessage())]);
        } catch (Exception $e) {
            $resultJson->setData(['success' => false, 'error' => true, 'message' => __($e->getMessage())]);
        }

        return $resultJson;
    }

    /**
     * @return \FiloBlu\Newsletter\Model\LandingPage
     */
    public function getLandingPage()
    {
        if (!$this->landingPage) {
            $this->landingPage = ObjectManager::getInstance()
                ->get(LandingPageFactory::class)
                ->create()
                ->load($this->getRequest()->getParam('landing_id'));
        }

        return $this->landingPage;
    }

    protected function allowExistingCustomers()
    {
        return (bool)$this->getLandingPage()->getAssignCustomerGroup();
    }

}
