<?php

/**
 *
 * Copyright © 2016 Magento. All rights reserved.
 * See COPYING.txt for license details.
 */

namespace FiloBlu\Newsletter\Controller\Subscriber;

use Exception;
use FiloBlu\Newsletter\Helper\Data;
use FiloBlu\Newsletter\Model\PreConfirm;
use FiloBlu\Newsletter\Model\ThankYou;
use Magento\Customer\Model\Session;
use Magento\Customer\Model\Url as CustomerUrl;
use Magento\Framework\App\Action\Context;
use Magento\Framework\Controller\Result\Json;
use Magento\Framework\Controller\Result\JsonFactory;
use Magento\Framework\Controller\Result\Redirect;
use Magento\Framework\Controller\ResultFactory;
use Magento\Framework\Controller\ResultInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\View\Result\Layout;
use Magento\Newsletter\Model\Subscriber;
use Magento\Newsletter\Model\SubscriberFactory;
use Magento\Store\Model\StoreManagerInterface;

/**
 * Class NewAction
 * @package FiloBlu\Newsletter\Controller\Subscriber
 */
class NewAction extends \Magento\Newsletter\Controller\Subscriber\NewAction
{
    /**
     * New subscription action
     *
     * @return Json|Redirect|ResultInterface|Layout|null
     */
    public function execute()
    {
        /** @var Data $helper */
        $helper = $this->_objectManager->get(Data::class);

        if ($helper->getConfig('popup/enabled') && $this->getRequest()->isPost() && $this->getRequest()->getPost('email')) {
            $resultJson = $this->resultFactory->create(ResultFactory::TYPE_JSON);

            $email = (string)$this->getRequest()->getPost('email');

            try {
                $this->validateEmailFormat($email);
                $this->validateGuestSubscription();
                $this->validateEmailAvailable($email);

                $subscriber = $this->_subscriberFactory->create();

                $this->_eventManager->dispatch('filoblu_newsletter_after_subscriber_created', [
                    'subscriber' => $subscriber,
                    'post_params' => $this->_request->getParams(),
                    'customer' => $this->_customerSession->getCustomer()
                ]);

                $status = $subscriber->subscribe($email);

                if ($status == Subscriber::STATUS_NOT_ACTIVE) {
                    $resultJson->setData([
                        'success' => true,
                        'error'   => false,
                        'message' => __('The confirmation request has been sent.')]);
                } else {
                    $resultJson->setData([
                        'success' => true,
                        'error'   => false,
                        'message' => __('Thank you for your subscription.')]);
                }
            } catch (LocalizedException $e) {
                $resultJson->setData([
                    'success' => false,
                    'error'   => true,
                    'message' => __('There was a problem with the subscription: %1', $e->getMessage())]);
            } catch (Exception $e) {
                $resultJson->setData([
                    'success' => false,
                    'error'   => true,
                    'message' => __($e->getMessage())]);
            }

            return $resultJson;
        }

        return null;
    }

    /**
     * Validates that the email address isn't being used by a different account.
     *
     * @param string $email
     * @return void
     * @throws LocalizedException
     */
    protected function validateEmailAvailable($email)
    {
        $websiteId = $this->_storeManager->getStore()->getWebsiteId();
        if ($this->_customerSession->getCustomerDataObject()->getEmail() !== $email && !$this->customerAccountManagement->isEmailAvailable($email, $websiteId)
        ) {
            throw new LocalizedException(
                __('This email address is already assigned to another user.')
            );
        }

        $subscriber = $this->_subscriberFactory->create()->loadByEmail($email);

        if ($subscriber->getSubscriberStatus() == Subscriber::STATUS_SUBSCRIBED) {
            throw new LocalizedException(
                __('This email address is already assigned to another user.')
            );
        }
    }
}
