<?php
/**
 * Copyright © 2016 Magento. All rights reserved.
 * See COPYING.txt for license details.
 */

namespace FiloBlu\Label\Model\Translate;

if(interface_exists(\Laminas\Filter\FilterInterface::class)) {
    interface FilterInterface extends \Laminas\Filter\FilterInterface
    {
    }
}else{
    interface FilterInterface extends \Zend_Filter_Interface
    {
    }
}

/**
 * This class is responsible for parsing content and applying necessary html element
 * wrapping and client scripts for inline translation.
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class Parser extends \Magento\Translation\Model\Inline\Parser implements \Magento\Framework\Translate\Inline\ParserInterface
{

    protected $_locale;

    /**
     * Initialize base inline translation model
     *
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     * @param \Magento\Translation\Model\ResourceModel\StringUtilsFactory $resource
     * @param \Zend_Filter_Interface $inputFilter
     * @param \Magento\Framework\App\State $appState
     * @param \Magento\Framework\App\Cache\TypeListInterface $appCache,
     * @param \Magento\Framework\Translate\InlineInterface $translateInline
     */
    public function __construct(
        \Magento\Translation\Model\ResourceModel\StringUtilsFactory $resource,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        FilterInterface $inputFilter,
        \Magento\Framework\App\State $appState,
        \Magento\Framework\App\Cache\TypeListInterface $appCache,
        \Magento\Framework\Translate\InlineInterface $translateInline,
        \Magento\Framework\Locale\Resolver $localeResolver
    ) {
        $this->_resourceFactory = $resource;
        $this->_storeManager = $storeManager;
        $this->_inputFilter = $inputFilter;
        $this->_appState = $appState;
        $this->_appCache = $appCache;
        $this->_translateInline = $translateInline;
        $this->_localeResolver = $localeResolver;
    }

    /**
     * Replace html body with translation wrapping.
     *
     * @param string $body
     * @return string
     */
    public function processResponseBodyString($body)
    {
        $this->_content = $body;

        $this->_specialTags();
        $this->_tagAttributes();
        $this->_otherText();

        return $this->_content;
    }

    /**
     * Prepare special tags
     *
     * @return void
     */
    private function _specialTags()
    {
        $this->_translateTags($this->_content, $this->_allowedTagsGlobal, '_applySpecialTagsFormat');
        $this->_translateTags($this->_content, $this->_allowedTagsSimple, '_applySimpleTagsFormat');
    }

    /**
     * Prepare tags inline translates
     *
     * @return void
     */
    private function _tagAttributes()
    {
        $this->_prepareTagAttributesForContent($this->_content);
    }

    /**
     * Prepare tags inline translates for the content
     *
     * @param string &$content
     * @return void
     */
    private function _prepareTagAttributesForContent(&$content)
    {
        $quoteHtml = $this->_getHtmlQuote();
        $tagMatch = [];
        $nextTag = 0;
        $tagRegExp = '#<([a-z]+)\s*?[^>]+?((' . self::REGEXP_TOKEN . ')[^>]*?)+\\\\?/?>#iS';
        while (preg_match($tagRegExp, $content, $tagMatch, PREG_OFFSET_CAPTURE, $nextTag)) {
            $tagHtml = $tagMatch[0][0];
            $matches = [];
            $attrRegExp = '#' . self::REGEXP_TOKEN . '#S';
            $trArr = $this->_getTranslateData($attrRegExp, $tagHtml, [$this, '_getAttributeLocation']);
            if ($trArr) {
                $transRegExp = '# ' . $this->_getHtmlAttribute(
                        self::DATA_TRANSLATE,
                        '\[([^' . preg_quote($quoteHtml) . ']*)]'
                    ) . '#i';
                if (preg_match($transRegExp, $tagHtml, $matches)) {
                    $tagHtml = str_replace($matches[0], '', $tagHtml);
                    $trAttr = ' ' . $this->_getHtmlAttribute(
                            self::DATA_TRANSLATE,
                            '[' . htmlspecialchars($matches[1]) . ',' . str_replace("\"", "'", join(',', $trArr)) . ']'
                        );
                } else {
                    $trAttr = ' ' . $this->_getHtmlAttribute(
                            self::DATA_TRANSLATE,
                            '[' . str_replace("\"", "'", join(',', $trArr)) . ']'
                        );
                }
                $trAttr = $this->_addTranslateAttribute($trAttr);

                $tagHtml = substr_replace($tagHtml, $trAttr, strlen($tagMatch[1][0]) + 1, 1);
                $content = substr_replace($content, $tagHtml, $tagMatch[0][1], strlen($tagMatch[0][0]));
            }
            $nextTag = $tagMatch[0][1] + strlen($tagHtml);
        }
    }

    /**
     * Get html element attribute
     *
     * @param string $name
     * @param string $value
     * @return string
     */
    private function _getHtmlAttribute($name, $value)
    {
        return $name . '=' . $this->_getHtmlQuote() . $value . $this->_getHtmlQuote();
    }

    /**
     * Add data-translate-mode attribute
     *
     * @param string $trAttr
     * @return string
     */
    private function _addTranslateAttribute($trAttr)
    {
        $translateAttr = $trAttr;
        $additionalAttr = $this->_getAdditionalHtmlAttribute();
        if ($additionalAttr !== null) {
            $translateAttr .= ' ' . $additionalAttr . ' ';
        }
        return $translateAttr;
    }

    /**
     * Get html quote symbol
     *
     * @return string
     */
    private function _getHtmlQuote()
    {
        if ($this->_isJson) {
            return '\"';
        } else {
            return '"';
        }
    }

    /**
     * Prepare simple tags
     *
     * @param string &$content
     * @param array $tagsList
     * @param string|array $formatCallback
     * @return void
     */
    private function _translateTags(&$content, $tagsList, $formatCallback)
    {
        $nextTag = 0;

        $tags = implode('|', array_keys($tagsList));
        $tagRegExp = '#<(' . $tags . ')(/?>| \s*[^>]*+/?>)#iSU';
        $tagMatch = [];
        while (preg_match($tagRegExp, $content, $tagMatch, PREG_OFFSET_CAPTURE, $nextTag)) {
            $tagName = strtolower($tagMatch[1][0]);
            if (substr($tagMatch[0][0], -2) == '/>') {
                $tagClosurePos = $tagMatch[0][1] + strlen($tagMatch[0][0]);
            } else {
                $tagClosurePos = $this->_findEndOfTag($content, $tagName, $tagMatch[0][1]);
            }

            if ($tagClosurePos === false) {
                $nextTag += strlen($tagMatch[0][0]);
                continue;
            }

            $tagLength = $tagClosurePos - $tagMatch[0][1];

            $tagStartLength = strlen($tagMatch[0][0]);

            $tagHtml = $tagMatch[0][0] . substr(
                    $content,
                    $tagMatch[0][1] + $tagStartLength,
                    $tagLength - $tagStartLength
                );
            $tagClosurePos = $tagMatch[0][1] + strlen($tagHtml);

            $trArr = $this->_getTranslateData(
                '#' . self::REGEXP_TOKEN . '#iS',
                $tagHtml,
                [$this, '_getTagLocation'],
                ['tagName' => $tagName, 'tagList' => $tagsList]
            );

            if (!empty($trArr)) {
                $trArr = array_unique($trArr);
                $tagHtml = call_user_func([$this, $formatCallback], $tagHtml, $tagName, $trArr);
                $tagClosurePos = $tagMatch[0][1] + strlen($tagHtml);
                $content = substr_replace($content, $tagHtml, $tagMatch[0][1], $tagLength);
            }
            $nextTag = $tagClosurePos;
        }
    }

    /**
     * Find end of tag
     *
     * @param string $body
     * @param string $tagName
     * @param int $from
     * @return bool|int return false if end of tag is not found
     */
    private function _findEndOfTag($body, $tagName, $from)
    {
        $openTag = '<' . $tagName;
        $closeTag = ($this->_isJson ? '<\\/' : '</') . $tagName;
        $tagLength = strlen($tagName);
        $length = $tagLength + 1;
        $end = $from + 1;
        while (substr_count($body, $openTag, $from, $length) !== substr_count($body, $closeTag, $from, $length)) {
            $end = strpos($body, $closeTag, $end + $tagLength + 1);
            if ($end === false) {
                return false;
            }
            $length = $end - $from + $tagLength + 3;
        }
        if (preg_match('#<\\\\?\/' . $tagName . '\s*?>#i', $body, $tagMatch, null, $end)) {
            return $end + strlen($tagMatch[0]);
        } else {
            return false;
        }
    }

    /**
     * Prepare other text inline translates
     *
     * @return void
     */
    private function _otherText()
    {
        $next = 0;
        $matches = [];
        while (preg_match('#' . self::REGEXP_TOKEN . '#', $this->_content, $matches, PREG_OFFSET_CAPTURE, $next)) {

            $label = [
                'shown' => $matches[1][0],
                'translated' => $matches[2][0],
                'original' => $matches[3][0],
                'location' => 'Text',
                'scope' => $matches[4][0],
            ];

            if (!empty($label)) {
                $params_to_save[] = [

                    'perstore' => true,
                    'original' => $label['original'],
                    'custom' => $label['translated']

                ];
            }


            $translateProperties = json_encode($label);

            $spanHtml = $this->_getDataTranslateSpan(
                '[' . htmlspecialchars($translateProperties) . ']',
                $matches[1][0]
            );

            $this->_content = substr_replace($this->_content, $spanHtml, $matches[0][1], strlen($matches[0][0]));
            $next = $matches[0][1] + strlen($spanHtml) - 1;
        }
//
//        if (!empty($params_to_save))
//            $this->processLabel($params_to_save);

    }

    /**
     * Get translate data by regexp
     *
     * @param string $regexp
     * @param string &$text
     * @param string|array $locationCallback
     * @param array $options
     * @return array
     */
    private function _getTranslateData($regexp, &$text, $locationCallback, $options = [])
    {
        $trArr = [];
        $next = 0;
        while (preg_match($regexp, $text, $matches, PREG_OFFSET_CAPTURE, $next)) {

            $label = [
                'shown' => htmlspecialchars_decode($matches[1][0]),
                'translated' => htmlspecialchars_decode($matches[2][0]),
                'original' => htmlspecialchars_decode($matches[3][0]),
                'location' => htmlspecialchars_decode(call_user_func($locationCallback, $matches, $options)),
                 ];

            if (!empty($label)) {
                $params_to_save[] = [

                    'perstore' => true,
                    'original' => $label['original'],
                    'custom' => $label['translated']

                ];
            }

            $trArr[] = json_encode($label);

            $text = substr_replace($text, $matches[1][0], $matches[0][1], strlen($matches[0][0]));
            $next = $matches[0][1];
        }
        if (!empty($params_to_save))
            $this->processLabel($params_to_save);


        return $trArr;
    }

    /**
     * Parse and save edited translation
     *
     * @param array $translateParams
     * @return array
     */
    public function processLabel(array $translateParams)
    {
        if (!$this->_translateInline->isAllowed()) {
            return ['inline' => 'not allowed'];
        }
        $this->_appCache->invalidate(\Magento\Framework\App\Cache\Type\Translate::TYPE_IDENTIFIER);

        $this->_validateTranslationParams($translateParams);
        $this->_filterTranslationParams($translateParams, ['custom']);

        /** @var $validStoreId int */
        $validStoreId = $this->_storeManager->getStore()->getId();


        foreach ($translateParams as $param) {
            if ($this->_appState->getAreaCode() != \Magento\Backend\App\Area\FrontNameResolver::AREA_CODE) {
                $this->saveTranslate($param['original'], $param['custom'], null, $validStoreId);
            }
        }
        $cacheManager = \Magento\Framework\App\ObjectManager::getInstance()->get(
            'Magento\Translation\Model\Inline\CacheManager'
        );

        return $cacheManager->updateAndGetTranslations();
    }

    /**
     * Save translation
     *
     * @param String $string
     * @param String $translate
     * @param String $locale
     * @param int|null $storeId
     * @return $this
     */
    public function saveTranslate($string, $translate, $locale = null, $storeId = null)
    {
        $string = htmlspecialchars_decode($string);
        /** @var $resource \Magento\Translation\Model\ResourceModel\StringUtils */
        $resource = $this->_resourceFactory->create();
        $connection = $resource->getConnection();
        $table = $resource->getMainTable();
        $translate = htmlspecialchars($translate, ENT_QUOTES);

        if ($locale === null) {
            $locale = $this->_localeResolver->getLocale();
        }

        if ($storeId === null) {
            $storeId = $this->getStoreId();
        }

        $select = $connection->select()->from(
            $table,
            ['key_id', 'translate']
        )->where(
            'store_id = :store_id'
        )->where(
            'locale = :locale'
        )->where(
            'string = :string'
        )->where(
            'crc_string = :crc_string'
        );
        $bind = [
            'store_id' => $storeId,
            'locale' => $locale,
            'string' => $string,
            'crc_string' => crc32($string),
        ];

        if ($row = $connection->fetchRow($select, $bind)) {
            if ($row['translate'] != $translate) {
                $connection->update($table, ['translate' => $translate], ['key_id=?' => $row['key_id']]);
            }
        } else {
            $connection->insert(
                $table,
                [
                    'store_id' => $storeId,
                    'locale' => $locale,
                    'string' => $string,
                    'translate' => $translate,
                    'crc_string' => crc32($string)
                ]
            );
        }

        return $this;
    }

}
